import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoErpUpdateDefectRateExecutable(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/organization/update_defect_rate/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="update_defect_rate")


class YdoErpUpdateDefectRate(sdk2.Task, SolomonMixinV2):
    """Task that updates organization defect rate info from uslugi"""

    class Parameters(generate_view(
        processor=YdoErpUpdateDefectRateExecutable,
        check_owner=True,
    )):
        ydo_intapi_host = sdk2.parameters.String('Ydo IntApi Host', required=True)

        tvm_id = sdk2.parameters.Integer('Ydo Tvm Id', required=True)

        tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

        with_ignoring_organization_without_partner_id = sdk2.parameters.Bool('Need for some organizations in testing', default=False)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing defect rate update')

        env = os.environ.copy()
        env['TVM_SECRET'] = self.Parameters.tvm_secret.data()['TVM_SECRET']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                '--ydo_intapi_host', str(self.Parameters.ydo_intapi_host),
                '--tvm_id', str(self.Parameters.tvm_id),
            ] + (['--with_ignoring_organization_without_partner_id'] if self.Parameters.with_ignoring_organization_without_partner_id else []),
            'update_defect_rate',
            'Failed to update defect rate',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpUpdateDefectRate, self).on_finish(prev_status, status)
