import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoErpUpdateEmployeesCalendarsProcessor(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/employees/update_employees_calendars/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="update_employees_calendars")


class YdoErpUpdateEmployeesCalendars(sdk2.Task, SolomonMixinV2):
    """Task that updates ERP employees calendars in YT for analytics"""

    class Parameters(
        generate_view(
            processor=YdoErpUpdateEmployeesCalendarsProcessor,
            check_owner=True,
        )
    ):
        ydo_oauth_tokens = sdk2.parameters.YavSecret("YAV YDO_OAUTH_TOKENS secret id", required=True)
        ydo_secrets = sdk2.parameters.YavSecret("YAV YDO_SECRETS secret id", required=True)

        ydb_host = sdk2.parameters.String('YDB Host', required=True)
        ydb_port = sdk2.parameters.Integer('YDB Port', default=2135, required=True)
        ydb_database = sdk2.parameters.String('YDB Database', required=True)
        ydb_home = sdk2.parameters.String('YDB Home', required=True)

        yt_cluster = sdk2.parameters.String('YT cluster', required=True, default='hahn')
        yt_folder = sdk2.parameters.String('YT folder', required=True)

        ydo_intapi_host = sdk2.parameters.String('YDO intapi host', required=True)
        tvm_id = sdk2.parameters.Integer('TVM id', required=True)

        days_qnt = sdk2.parameters.Integer('Total days count', default=14, required=True)
        days_in_past_qnt = sdk2.parameters.Integer('Days count in the past', default=0, required=True)
        batch_size = sdk2.parameters.Integer('Batch size', default=1000, required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (PipEnvironment('requests'),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing update employees calendars processor')

        env = os.environ.copy()
        env['YDB_TOKEN'] = self.Parameters.ydo_oauth_tokens.data()['YDB_TOKEN']
        env['YT_TOKEN'] = self.Parameters.ydo_oauth_tokens.data()['YT_TOKEN']
        env['TVM_SECRET'] = self.Parameters.ydo_secrets.data()['TVM_SECRET']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                '--ydb_host',
                str(self.Parameters.ydb_host),
                '--ydb_port',
                str(self.Parameters.ydb_port),
                '--ydb_database',
                str(self.Parameters.ydb_database),
                '--ydb_home',
                str(self.Parameters.ydb_home),
                '--yt_cluster',
                str(self.Parameters.yt_cluster),
                '--yt_folder',
                str(self.Parameters.yt_folder),
                '--ydo_intapi_host',
                str(self.Parameters.ydo_intapi_host),
                '--tvm_id',
                str(self.Parameters.tvm_id),
                '--days_qnt',
                str(self.Parameters.days_qnt),
                '--days_in_past_qnt',
                str(self.Parameters.days_in_past_qnt),
                '--batch_size',
                str(self.Parameters.batch_size),
            ],
            'update_employees_calendars',
            'Failed to update employees calendars',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpUpdateEmployeesCalendars, self).on_finish(prev_status, status)
