import json
import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoErpServicesSchemasJson(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoErpServicesSchemasMergerExecutable(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/erp/tools/services/services_schemas/generate_services_schemas/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="generate_services_schemas")


class YdoErpServicesSchemasMerger(nanny.ReleaseToNannyTask2, sdk2.Task, SolomonMixinV2):
    """Task that converts partner rubricator to ERP format"""

    class Parameters(sdk2.Parameters):
        merger_resource = sdk2.parameters.Resource(
            'merger',
            resource_type=YdoErpServicesSchemasMergerExecutable,
            required=True,
        )

        services_schemas_resource = sdk2.parameters.Resource(
            'services_schemas',
            resource_type=YdoErpServicesSchemasJson,
            required=True,
        )

        ydo_intapi_host = sdk2.parameters.String('Ydo IntApi Host', required=True)

        tvm_id = sdk2.parameters.Integer('Ydo Tvm Id', required=True)

        tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

        auto_release_policy = sdk2.parameters.RadioGroup(
            'Auto Release Policy',
            choices=(
                ('force', 'force'),
                ('if_changed', 'if_changed'),
            ),
            required=True,
            default='if_changed',
        )

        regard_inexact = sdk2.parameters.Bool('Regard inexact services', default=False)

        additional_services_cloning_policy = sdk2.parameters.RadioGroup(
            'Additional services cloning policy',
            choices=(
                ('do_not_clone', 'do_not_clone'),
                ('clone_on_subtree', 'clone_on_subtree'),
                ('clone_on_last_layer', 'clone_on_last_layer'),
            ),
            required=True,
            default='clone_on_subtree',
        )

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing merger')

        services_schemas_resource = sdk2.ResourceData(YdoErpServicesSchemasJson(self, 'Ydo erp services_schemas.json', 'services_schemas.json'))

        env = os.environ.copy()
        env['TVM_SECRET'] = self.Parameters.tvm_secret.data()['TVM_SECRET']

        execute_cmd(
            [
                str(sdk2.ResourceData(self.Parameters.merger_resource).path),
                '--ydo_intapi_host', self.Parameters.ydo_intapi_host,
                '--tvm_id', str(self.Parameters.tvm_id),
                '--output', 'services_schemas.json',
                '--additional_services_cloning_policy', str(self.Parameters.additional_services_cloning_policy),
            ]
            + (['--regard_inexact'] if self.Parameters.regard_inexact else []),
            'generate_services_schemas',
            'Failed to convert partner rubricator to ERP format',
            env=env,
        )

        services_schemas_resource.ready()

        can_auto_release = True
        if self.Parameters.auto_release_policy == 'if_changed':
            with open(str(sdk2.ResourceData(self.Parameters.services_schemas_resource).path), 'r') as released_services_schemas, open('services_schemas.json', 'r') as current_services_schemas:
                if json.load(released_services_schemas) == json.load(current_services_schemas):
                    can_auto_release = False

        self.Context.can_auto_release = can_auto_release
        self.Context.save()

        logging.info('Done')

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoErpServicesSchemasMerger, self).on_finish(prev_status, status)
