# coding: utf-8
import logging

import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common import task_env
from sandbox.projects.ydo import ydo_releasers
from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class YdoErpWorkersCollectorExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoErpWorkersCollector(sdk2.Task):
    """
        Сбор информации из erp/employees в external_workers
    """

    class Requirements(task_env.TinyRequirements):
        ram = 1024
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(generate_view(ydo_erp_workers_collector=YdoErpWorkersCollectorExecutable, check_owner=True, with_environ=True)):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            erp_employees_table = sdk2.parameters.String('Path to erp employees backup', required=True)
            erp_organizations_table = sdk2.parameters.String('Path to erp organizations backup', required=True)
            external_workers_table = sdk2.parameters.String('Path to external_workers backup', required=True)
        logging_config = sdk2.parameters.String('LoggingConfig')

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        cmd = [
            self.Parameters.ydo_erp_workers_collector_path,
            '--kikimr-host', self.Parameters.ydb_host,
            '--kikimr-home', self.Parameters.ydb_home,
            '--kikimr-database', self.Parameters.ydb_database,
            '--kikimr-port', str(self.Parameters.ydb_port),

            '--erp_employees', self.Parameters.erp_employees_table,
            '--erp_organizations', self.Parameters.erp_organizations_table,
            '--external_workers', self.Parameters.external_workers_table,
        ]
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))
        execute_cmd(cmd, 'ydo_erp_workers_collector', 'ydo_erp_workers_collector failed', env=self.Parameters.get_environ())
