import os
import json

from sandbox import sdk2, common
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskError
from sandbox.projects.common.nanny import nanny
from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoFeaturesJson
)


class YdoGenerateFeaturesJsonExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoGenerateFeaturesJson(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Parameters):
        dump_to_json = sdk2.parameters.Resource(
            'dump_to_json executable',
            resource_type=YdoGenerateFeaturesJsonExecutable,
            required=True
        )

        with sdk2.parameters.Group('YDB settings') as ydb_settings_block:
            ydb_host = sdk2.parameters.String('ydb_host', required=True)
            ydb_port = sdk2.parameters.String('ydb_port', required=True)
            ydb_database = sdk2.parameters.String('ydb_database', required=True)
            ydb_home = sdk2.parameters.String('ydb_home', required=True)

        with sdk2.parameters.Group('YT settings') as yt_settings_block:
            yt_proxy = sdk2.parameters.String('yt_proxy', required=True)
            yt_path = sdk2.parameters.String('dump yt_path')

        with sdk2.parameters.RadioGroup('Compare to last release') as compare_to:
            compare_to.values['stable'] = compare_to.Value('Stable', default=True)
            compare_to.values['testing'] = compare_to.Value('Testing')
            compare_to.values['none'] = compare_to.Value('No validation')

        additional_cmd_params = sdk2.parameters.String('Additional cmd args')

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        resource = YdoFeaturesJson(self, 'generated features.json', 'features.json')
        resource_data = sdk2.ResourceData(resource)

        yt_token = sdk2.Vault.data(self.owner, 'yt-token')
        yt_proxy = self.Parameters.yt_proxy

        env = os.environ.copy()
        env['YDB_TOKEN'] = sdk2.Vault.data(self.owner, 'ydb-token')
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = yt_proxy

        cmd_params = [
            str(sdk2.ResourceData(self.Parameters.dump_to_json).path),
            '--ydb_host', self.Parameters.ydb_host,
            '--ydb_port', self.Parameters.ydb_port,
            '--ydb_database', self.Parameters.ydb_database,
            '--ydb_home', self.Parameters.ydb_home,
            '--output', str(resource_data.path)
        ]
        if self.Parameters.yt_path:
            cmd_params.extend([
                '--table', self.Parameters.yt_path
            ])
        if self.Parameters.compare_to != 'none':
            cmd_params.extend([
                '--compare', self.Parameters.compare_to,
                '--validation-output', 'validation_output.json'
            ])
        if self.Parameters.additional_cmd_params:
            cmd_params.extend(self.Parameters.additional_cmd_params.split())

        try:
            execute_cmd(cmd_params, 'generate_features_json', fail_on_error=False, env=env)
        except TaskError as exception:
            try:
                validation_errors = json.load(open('validation_output.json', 'r'))
                if validation_errors:
                    self.set_info(
                        'Validation errors<ul>{}</ul>'.format(''.join([
                            '<li>{}</li>'.format(err) for err in validation_errors
                        ])),
                        do_escape=False
                    )
            except IOError:
                pass
            raise exception

        resource_data.ready()

        self.Context.new_features_data = open(str(resource_data.path), 'r').read()
        self.Context.save()

    def on_release(self, parameters):
        last_features_data = self.get_last_features_data(parameters['release_status'])

        if last_features_data == self.Context.new_features_data:
            self.set_info('Same as the last release')
        else:
            nanny.ReleaseToNannyTask2.on_release(self, parameters)
            sdk2.Task.on_release(self, parameters)

    def get_last_features_data(self, release_status):
        last_resource = YdoFeaturesJson.find(attrs=dict(released=release_status)).first()
        if last_resource is not None:
            last_resource_data = sdk2.ResourceData(last_resource)
            return open(str(last_resource_data.path), 'r').read()
        return None
