import logging
import os

from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    get_now_utc,
    execute_cmd,
)

from sandbox.projects.ydo.ferryman import (
    start_ferryman_process,
    wait_ferryman,
)

from sandbox.projects.geosearch.CleanupYtFolder import clean_folder

from sandbox.sandboxsdk import environments


# Resource types
class YdoMakeKvForFeedbackExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoMakeKvForFeedback(sdk2.Task):
    class Parameters(sdk2.Parameters):
        kv_for_feedback = sdk2.parameters.Resource(
            'Make kv for feedback executable',
            resource_type=YdoMakeKvForFeedbackExecutable,
            required=True
        )

        upload_to_stable = sdk2.parameters.Bool("Upload to stable", default=False)
        saas_namespace = sdk2.parameters.String('SAAS namespace (kps)', required=True)
        yt_source_path = sdk2.parameters.String('Path to table with analytics hypotesis', required=True)
        yt_res_dir = sdk2.parameters.String('Path to result table', required=True)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def create_kv_table(self):
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = 'hahn.yt.yandex.net'
        env['YT_LOG_LEVEL'] = 'INFO'

        self.Context.index_ts = get_now_utc()
        self.Context.out_table = os.path.join(self.Parameters.yt_res_dir, '{mode}/{ts}_kps={kps}'.format(
            ts=self.Context.index_ts, mode='stable' if self.Parameters.upload_to_stable else 'prestable', kps=self.Parameters.saas_namespace))

        logging.info('Start making yt table for kv...')
        execute_cmd(
            [
                str(sdk2.ResourceData(self.Parameters.kv_for_feedback).path),
                '--source_path', self.Parameters.yt_source_path,
                '--res_path', self.Context.out_table,
            ],
            'kv_for_feedback',
            'Failed to create kv',
            env=env
        )

    def on_execute(self):
        with self.memoize_stage.firststage:
            self.create_kv_table()
            self.Context.ferryman_name = 'ydo-review-candidates' + ('' if self.Parameters.upload_to_stable else '-prestable')
            self.Context.ferryman_batch_id = start_ferryman_process({self.Context.out_table: self.Parameters.saas_namespace}, self.Context.index_ts,
                                                                    self.Context.ferryman_name, sdk2.Vault.data(self.owner, 'yt-token'))

        with self.memoize_stage.wait_ferryman(10):
            wait_ferryman(self.Context.ferryman_name, self.Context.ferryman_batch_id)

        with self.memoize_stage.clean_folder:
            path_to_clean = os.path.join(self.Parameters.yt_res_dir, 'stable' if self.Parameters.upload_to_stable else 'prestable')
            clean_folder(self, path_to_clean)

        logging.info('Done')
