import json
import logging
import requests
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure

TRY_COUNT = 20
SLEEP_TIME = 30

FERRYMAN_URL_TEMPLATE = 'http://{ferryman_name}.ferryman.n.yandex-team.ru'

def generate_ferryman_request_params(out_table2saas_namespace, timestamp, cluster, proto):
    common_part_table = {
        'Timestamp': timestamp * 10**6,
        'Delta': False,
    }

    if cluster:
        common_part_table.update({'Cluster': cluster})

    if proto:
        common_part_table.update({'Format': 'message'})

    ferryman_request = []
    saas_namespaces = out_table2saas_namespace.values()
    for out_table, saas_namespace in out_table2saas_namespace.iteritems():
        current_table = {
            'Path': out_table,
            'Namespace': saas_namespace,
        }
        current_table.update(common_part_table)
        ferryman_request.append(current_table)

    return ferryman_request


def check_row_count(ferryman_request, ferryman_name, yt_token, cluster='hahn', degrade_level=1.0):
    import yt.wrapper as yt
    yt.config['token'] = yt_token
    yt.config['proxy']['url'] = cluster

    ferryman_url = FERRYMAN_URL_TEMPLATE.format(ferryman_name=ferryman_name)

    namespaces_status = {}

    for i in xrange(TRY_COUNT):
        try:
            namespaces_status_req = requests.get(ferryman_url + '/get-namespaces')
            namespaces_status_req.raise_for_status()
            namespaces_status = namespaces_status_req.json()
            break
        except requests.exceptions.HTTPError:
            logging.exception('Ferryman status request failed')
            if i + 1 == TRY_COUNT:
                raise
            time.sleep(SLEEP_TIME)

    filtered_infos = {info['namespace']: info for info in namespaces_status.get('final', [])}

    for table_node in ferryman_request:
        current_row_count = filtered_infos.get(table_node['Namespace'], {}).get('rowCount', 0)
        new_row_count = yt.row_count(table_node['Path'])
        if current_row_count != 0 and float(current_row_count - new_row_count) / current_row_count > degrade_level:
            raise TaskFailure('Table {} (kps={}) is too small: expected {} rows (-{}%), got {}'.format(table_node['Path'], table_node['Namespace'], current_row_count, degrade_level * 100, new_row_count))


def get_add_full_tables_params(ferryman_name, ferryman_request):
    ferryman_url = FERRYMAN_URL_TEMPLATE.format(ferryman_name=ferryman_name)
    params = {'tables': json.dumps(ferryman_request, separators=(',', ':'))}

    return ferryman_url + '/add-full-tables', params


def send_tables_to_ferryman(ferryman_name, ferryman_request):
    ferryman_answer = {}

    logging.info('Trying to upload index...')
    ferryman_url, params = get_add_full_tables_params(ferryman_name, ferryman_request)
    for i in xrange(TRY_COUNT):
        logging.info('Try #{}'.format(i))
        try:
            add_tables_req = requests.get(ferryman_url, params=params)
            add_tables_req.raise_for_status()

            ferryman_answer = add_tables_req.json()
            logging.info('Ferryman answer: %s', ferryman_answer)
            if add_tables_req.status_code != 202:
                raise TaskFailure('Bad ferryman request, details: {}'.format(ferryman_answer))
            break
        except TaskFailure:
            raise
        except Exception as e:
            logging.exception('Ferryman request failed')
            content = ''
            if add_tables_req is not None and add_tables_req.content:
                content = add_tables_req.content
                logging.error(add_tables_req.content)

            if i + 1 == TRY_COUNT:
                import sys
                raise type(e), type(e)(str(e) + '. Content: %s' % content), sys.exc_info()[2]

            time.sleep(SLEEP_TIME)
    ferryman_batch_id = ferryman_answer['batch']
    logging.info('Ferryman now uploading batch {}'.format(ferryman_batch_id))
    return ferryman_batch_id


def start_ferryman_process(out_table2saas_namespace, timestamp, ferryman_name, yt_token, cluster='hahn', degrade_level=1.0, proto=False):
    ferryman_request = generate_ferryman_request_params(out_table2saas_namespace, timestamp, cluster, proto)
    check_row_count(ferryman_request, ferryman_name, yt_token, cluster, degrade_level)
    ferryman_batch_id = send_tables_to_ferryman(ferryman_name, ferryman_request)
    return ferryman_batch_id


def wait_ferryman(ferryman_name, ferryman_batch_id, time_to_wait=25 * 60, target_statuses=None):
    logging.info('Waiting for ferryman...')
    params = {'batch': ferryman_batch_id}
    target_statuses = target_statuses or ('final', 'searchable')
    response = {}
    try:
        wait_req = requests.get('http://{ferryman_name}.ferryman.n.yandex-team.ru/get-batch-status'.format(
                         ferryman_name=ferryman_name), params=params)
        wait_req.raise_for_status()
        response = wait_req.json()
    except:
        pass
    if response.get('status') == 'error' or 'invalid_input' in response:
        raise TaskFailure('Error while ferryman request: {}'.format(response))
    if response.get('status') not in target_statuses:
        raise sdk2.WaitTime(time_to_wait)


def get_full_request_url(ferryman_request, ferryman_name):
    ferryman_url, params = get_add_full_tables_params(ferryman_name, ferryman_request)

    import urllib
    return '{}?{}'.format(ferryman_url, urllib.urlencode(params))
