import logging
import os
import tempfile

import sandbox.common.errors as ce

from sandbox import sdk2

import sandbox.projects.common.arcadia.sdk as arcadia_sdk

from sandbox.projects.common import file_utils as fu
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.ydo import execute_cmd, ydo_releasers, YdoOrderCustomizerJson
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump


class YdoGetLaunchedCategoriesExecutable(sdk2.Resource):
    """
    Merge launched categories from arcadia, flags.json and AB
    """

    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoGetLaunchedCategories(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(
        generate_view(
            ydo_get_launched_categories=YdoGetLaunchedCategoriesExecutable,
            geobase=GEODATA6BIN_STABLE,
            rubrics_json=YdoMergedRubricsSmallDump,
            order_clarifier_config=YdoOrderCustomizerJson,
            with_environ=True,
        )
    ):
        ab_url = sdk2.parameters.String('AB url', default='https://ab.yandex-team.ru')
        ab_all_categories_testid = sdk2.parameters.String('AB testid with all categories')
        out_path = sdk2.parameters.String('Path to table with results', required=True)
        out_hr_path = sdk2.parameters.String('Path to table with results in human-readable format', required=True)
        yt_proxy = sdk2.parameters.String('YT proxy', default='hahn')

    def on_execute(self):
        arc_token = None
        try:
            arc_token = sdk2.Vault.data(self.owner, 'ARC_TOKEN')
        except ce.VaultNotFound:
            pass
        additional_mount_kwargs = {}
        if arc_token:
            additional_mount_kwargs = dict(
                use_arc_instead_of_aapi=True,
                arc_oauth_token=arc_token,
            )
            logging.info('Use arc instead of aapi')

        arcadia_config_content = None
        flags_json_content = None
        with arcadia_sdk.mount_arc_path('arcadia:/arc/trunk/arcadia/ydo', **additional_mount_kwargs) as arcadia_path:
            with open(os.path.join(arcadia_path, 'rubricator/service_model_categories.json')) as arcadia_config:
                arcadia_config_content = arcadia_config.read()
            with open(os.path.join(arcadia_path, 'exp_flags/flags.json')) as flags_json:
                flags_json_content = flags_json.read()

        arcadia_config_path = tempfile.NamedTemporaryFile().name
        fu.write_file(arcadia_config_path, arcadia_config_content)

        flags_json_path = tempfile.NamedTemporaryFile().name
        fu.write_file(flags_json_path, flags_json_content)

        cmd = [
            self.Parameters.ydo_get_launched_categories_path,
            '--input_config_path',
            arcadia_config_path,
            '--flags_json_path',
            flags_json_path,
            '--ab_url',
            self.Parameters.ab_url,
            '--out_path',
            self.Parameters.out_path,
            '--out_hr_path',
            self.Parameters.out_hr_path,
            '--geobase_path',
            self.Parameters.geobase_path,
            '--rubrics_json_path',
            self.Parameters.rubrics_json_path,
            '--clarifier_config_path',
            self.Parameters.order_clarifier_config_path,
            '--yt_proxy',
            self.Parameters.yt_proxy,
        ]
        if self.Parameters.ab_all_categories_testid:
            cmd.extend(['--ab_all_categories_testid', self.Parameters.ab_all_categories_testid])

        execute_cmd(cmd, 'get_launched_categories', 'get_launched_categories failed', env=self.Parameters.get_environ())
