import datetime

from sandbox import sdk2
from sandbox.projects.ydo import ydo_releasers

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoIncidentsObserverExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoIncidentsObserver(sdk2.Task, SolomonMixinV2):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(generate_view(ydo_incidents_observer=YdoIncidentsObserverExecutable, check_owner=True, with_environ=True)):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            yt_puid_to_worker_id = sdk2.parameters.String('YT puid_to_worker_id path', required=True)
            yt_incidents = sdk2.parameters.String('YT incidents path', required=True)
            yt_incident_triggers = sdk2.parameters.String('YT incident_triggers path', required=True)
            yt_workers = sdk2.parameters.String('YT workers path', required=True)
            yt_complaints = sdk2.parameters.String('YT complaints path', required=True)
            yt_connection_map = sdk2.parameters.String('YT connection map path', default='')
            yt_antispam_complaints = sdk2.parameters.String('YT antispam complaints with verdicts path', default='')
            yt_puid2antifraud_glue_id = sdk2.parameters.String('YT puid2antifraud_glue_id path', required=True)
            yt_one_table_complaints = sdk2.parameters.Bool('Use exactly complaints parameter as complaints table path', default=False)
            yt_deleted_workers = sdk2.parameters.String('YT deleted workers path', required=True)
            yt_verified_reviews_scorings = sdk2.parameters.String('YT verified_reviews_scorings path', required=False)
            yt_orders_deception = sdk2.parameters.String('YT orders_deception path', required=False)
        with sdk2.parameters.Group('TRIGGERS'):
            triggers_ugc_reviews = sdk2.parameters.Bool('enable trigger from ugc reviews', default=True)
            triggers_fraud_complaints = sdk2.parameters.Bool('enable trigger from fraud complaints', default=True)
            triggers_antifraud_glues = sdk2.parameters.Bool('enable trigger from antifraud_glue', default=False)
            triggers_order_price_deception = sdk2.parameters.Bool('enable trigger from order_price_deception', default=False)
        with sdk2.parameters.Group('TRACKER'):
            tracker_queue_id = sdk2.parameters.String('Queue id for tracker issues', required=True)
            tracker_org_id = sdk2.parameters.String('Organization id for tracker', required=True)
            tracker_errors_threshold = sdk2.parameters.Float('Errors threshold for tracker issues creation (from 0 to 1)', default=0.2)
            max_tracker_issues_to_create = sdk2.parameters.Integer('Max tracker issues to create', required=True)
            update_tracker_issues = sdk2.parameters.Bool('enable update incidents from tracker issues', default=False)
        with sdk2.parameters.Group('BLACKBOX'):
            tvm_id = sdk2.parameters.Integer('TVM id')
            tvm_token_name = sdk2.parameters.String('TVM token env variable name')
            blackbox_env = sdk2.parameters.String('Blackbox env', choices=[('test', 'test'), ('mimino', 'mimino'), ('prod', 'prod')])

        logging_config = sdk2.parameters.String('LoggingConfig')
        max_workers_to_create_auto_approved_strikes = sdk2.parameters.Integer('Max workers to create auto approved strikes')
        from_date = sdk2.parameters.String('FromDate in a format "YYYY-mm-dd"')
        deception_from_date = sdk2.parameters.String('FromDate for price order deception in a format "YYYY-mm-dd"')

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        cmd = [
            self.Parameters.ydo_incidents_observer_path,
            '--kikimr-host', self.Parameters.ydb_host,
            '--kikimr-home', self.Parameters.ydb_home,
            '--kikimr-database', self.Parameters.ydb_database,
            '--kikimr-port', str(self.Parameters.ydb_port),

            '--puid_to_worker_id', self.Parameters.yt_puid_to_worker_id,
            '--incidents', self.Parameters.yt_incidents,
            '--incident_triggers', self.Parameters.yt_incident_triggers,
            '--complaints', self.Parameters.yt_complaints,
            '--workers', self.Parameters.yt_workers,
            '--puid2antifraud_glue_id', self.Parameters.yt_puid2antifraud_glue_id,
            '--deleted_workers',  self.Parameters.yt_deleted_workers,

            '--tracker_errors_threshold', str(self.Parameters.tracker_errors_threshold),
            '--tracker_queue_id', str(self.Parameters.tracker_queue_id),
            '--tracker_org_id', str(self.Parameters.tracker_org_id),
            '--max_tracker_issues_to_create', str(self.Parameters.max_tracker_issues_to_create),
        ]
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))
        self.add_param_to_cmd(cmd, '--from-date', self.Parameters.from_date or datetime.datetime.utcnow().strftime('%Y-%m-%d'))
        self.add_param_to_cmd(cmd, '--deception-from-date', self.Parameters.deception_from_date or (datetime.datetime.utcnow() - datetime.timedelta(days=1)).strftime('%Y-%m-%d'))

        self.add_param_to_cmd(cmd, '--tvm_id', str(self.Parameters.tvm_id))
        self.add_param_to_cmd(cmd, '--tvm_token_name', str(self.Parameters.tvm_token_name))
        self.add_param_to_cmd(cmd, '--blackbox_env', str(self.Parameters.blackbox_env))
        self.add_param_to_cmd(cmd, '--max_workers_to_create_auto_approved_strikes', str(self.Parameters.max_workers_to_create_auto_approved_strikes))
        self.add_param_to_cmd(cmd, '--verified_reviews_scorings', self.Parameters.yt_verified_reviews_scorings)
        self.add_param_to_cmd(cmd, '--orders_deception', self.Parameters.yt_orders_deception)
        self.add_param_to_cmd(cmd, '--antispam_complaints', self.Parameters.yt_antispam_complaints)
        self.add_param_to_cmd(cmd, '--connection_map', self.Parameters.yt_connection_map)

        if self.Parameters.triggers_ugc_reviews:
            cmd += ['--ugc_reviews']
        if self.Parameters.triggers_fraud_complaints:
            cmd += ['--fraud_complaints']
        if self.Parameters.triggers_antifraud_glues:
            cmd += ['--antifraud_glues']
        if self.Parameters.triggers_order_price_deception:
            cmd += ['--order_price_deception']
        if self.Parameters.yt_one_table_complaints:
            cmd += ['--one_table_complaints']
        if self.Parameters.update_tracker_issues:
            cmd += ['--update_tracker_issues']

        execute_cmd(cmd, 'ydo_incidents_observer', 'ydo_incidents_observer failed', env=self.Parameters.get_environ())

    def on_break(self, *args, **kwargs):
        SolomonMixinV2.on_break(self, *args, **kwargs)
        super(YdoIncidentsObserver, self).on_break(*args, **kwargs)

    def on_finish(self, *args, **kwargs):
        SolomonMixinV2.on_finish(self, *args, **kwargs)
        super(YdoIncidentsObserver, self).on_finish(*args, **kwargs)
