import logging
import os

from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
)


# Resource types
class YdoIndexerPrepareFactorsExecutable(sdk2.Resource):
    """
        Generate factors for SAAS index
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoIndexerPrepareFactors(sdk2.Task):
    '''Task that collect factors for SAAS index and write it to YT'''

    class Parameters(sdk2.Parameters):
        factors_preparer = sdk2.parameters.Resource(
            'indexer prepare factors executable',
            resource_type=YdoIndexerPrepareFactorsExecutable,
            required=True
        )

        auxiliary_path = sdk2.parameters.String('Path to dir with auxiliary factors', required=False)

        auxiliary_path_by_card = sdk2.parameters.String('Path to dir with auxiliary factors by card', required=False)

        actualization_path = sdk2.parameters.String('Path to actualization table', required=True)

        matching_path = sdk2.parameters.String('Path to matching table', required=False)

        phone_proxy_path = sdk2.parameters.String('Path to phone_proxy table', required=True)

        profiru_path = sdk2.parameters.String('Path to profiru ratings table', required=True)

        ugc_path = sdk2.parameters.String('Path to uslugi ratings table', required=True)

        verified_reviews_path = sdk2.parameters.String('Path to uslugi verified ratings table', required=False)

        verified_reviews_threshold = sdk2.parameters.Float('Threshold for computing verified ratings', default=0.5)

        next_version_verified_reviews_path = sdk2.parameters.String('Path to uslugi next version verified ratings table', required=False)

        next_version_verified_reviews_threshold = sdk2.parameters.Float('Threshold for computing next version verified ratings', default=0.5)

        youdo_path = sdk2.parameters.String('Path to youdo ratings table', required=True)

        db_dump_dir = sdk2.parameters.String('Path to db backup', required=False)

        activity_overall_data_path = sdk2.parameters.String('Path to raw activity data table for all time', required=False)

        activity_month_data_path = sdk2.parameters.String('Path to raw activity data table for the last 30 days', required=False)

        new_activity_data_path = sdk2.parameters.String('Path to directory with offline calculated activity data table for the last 4 months', required=False)

        res_path = sdk2.parameters.String('Path to resulting table', required=True)

        res_path_by_card = sdk2.parameters.String('Path to resulting table for card factors', required=False)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def prepare_data(self):
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = 'hahn.yt.yandex.net'
        env['YT_LOG_LEVEL'] = 'INFO'

        logging.info('Starting data preparation...')
        execute_cmd(
            [
                str(sdk2.ResourceData(self.Parameters.factors_preparer).path),
                '--actualization_path', self.Parameters.actualization_path,
                '--phone_proxy_path', self.Parameters.phone_proxy_path,
                '--profiru_path', self.Parameters.profiru_path,
                '--youdo_path', self.Parameters.youdo_path,
                '--res_path', self.Parameters.res_path,
                '--ugc_path', self.Parameters.ugc_path,
                '--db_dump_dir', self.Parameters.db_dump_dir,
            ]
            + (['--matching_path', self.Parameters.matching_path] if self.Parameters.matching_path else [])
            + (['--auxiliary_factors_path', self.Parameters.auxiliary_path] if self.Parameters.auxiliary_path else [])
            + (['--auxiliary_factors_path_by_card', self.Parameters.auxiliary_path_by_card] if self.Parameters.auxiliary_path_by_card else [])
            + (['--activity_overall_data_path', self.Parameters.activity_overall_data_path] if self.Parameters.activity_overall_data_path else [])
            + (['--activity_month_data_path', self.Parameters.activity_month_data_path] if self.Parameters.activity_month_data_path else [])
            + (['--new_activity_data_path', self.Parameters.new_activity_data_path] if self.Parameters.new_activity_data_path else [])
            + (['--verified_reviews_path', self.Parameters.verified_reviews_path,
                '--verified_reviews_threshold', str(self.Parameters.verified_reviews_threshold)] if self.Parameters.verified_reviews_path else [])
            + (['--next_version_verified_reviews_path', self.Parameters.next_version_verified_reviews_path,
                '--next_version_verified_reviews_threshold', str(self.Parameters.next_version_verified_reviews_threshold)] if self.Parameters.next_version_verified_reviews_path else [])
            + (['--res_path_by_card', self.Parameters.res_path_by_card] if self.Parameters.res_path_by_card else []),
            'ydo_indexer_prepare_factors',
            'Failed prepare factors for index',
            env=env
        )

    def on_execute(self):
        with self.memoize_stage.firststage:
            self.prepare_data()

        logging.info('Done')
