# -*- coding: utf-8 -*-

import logging
import os
import datetime

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.common.types import task as ctt
from sandbox.projects.ydo import execute_cmd

from sandbox.projects.ydo.resource_types import (
    YdoAnnSessionsPreparer,
    YdoAnnStatCalculator,
)

from sandbox.projects.ydo.indexer.YdoRunYQL2 import YdoRunYQL2


class YdoIndexAnn(sdk2.Task):
    '''Task that creates ydo annotations'''

    class Parameters(sdk2.Parameters):

        sessions_preparer = sdk2.parameters.Resource(
            'Sessions preparer',
            resource_type=YdoAnnSessionsPreparer,
            required=True
        )

        stat_calculator = sdk2.parameters.Resource(
            'Statistics calculator',
            resource_type=YdoAnnStatCalculator,
            required=True
        )

        birzha_ann_days_num = sdk2.parameters.Integer('Squeeze days for birzha annotations, leave empty for 90', default = 90)

        calc_birzha_ann = sdk2.parameters.Bool('Calculate birzha annotations')

        calc_birzha_quorum_ann = sdk2.parameters.Bool('Calculate birzha quorum annotations')

        end_date = sdk2.parameters.String('Squeeze end date, leave empty for now() - 3')

        days_num = sdk2.parameters.Integer('Squeeze days count', required=True)

        out_table = sdk2.parameters.String('Output table', required=True)

        external_ann = sdk2.parameters.String('External quorum annotations', required=False)

        quorum_ann = sdk2.parameters.Bool('Calc quorum annotations', default=True)

        quorum_mode = sdk2.parameters.String('Quorum ann mode', required=False)

        use_profile_clicks = sdk2.parameters.Bool('Join profile clicks to search sessions', default=False)

        real_region = sdk2.parameters.Bool('Use real region instead of 225', default=False)

        no_promo = sdk2.parameters.Bool('Ignore clicks from promo', default=False)

        cut_stopwords = sdk2.parameters.Bool('Cut stopwords from texts (use doppnorm)', default=False)

        proto_out = sdk2.parameters.Bool('Proto ann output', default=False)

        yt_log_level = sdk2.parameters.String('Yt log level', required=False)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        import yt.wrapper as yt
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')
        yt_proxy = 'hahn.yt.yandex.net'

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = yt_proxy
        env['YT_LOG_LEVEL'] = self.Parameters.yt_log_level if self.Parameters.yt_log_level else 'INFO'

        temp_reactions_table = "//home/ydo-dev/tmp/prepare_reactions_" + str(self.Parameters.end_date) + "_task_" + str(self.id)

        client = yt.YtClient(proxy=yt_proxy, token=yt_token)

        if self.Parameters.calc_birzha_ann:
            with self.memoize_stage.prepare_reactions:
                if self.Parameters.end_date:
                    react_ann_end_date = datetime.datetime.strptime(self.Parameters.end_date, "%Y%m%d").strftime('%Y-%m-%d')
                else:
                    react_ann_end_date = (datetime.datetime.now() - datetime.timedelta(days=3)).strftime('%Y-%m-%d')

                react_ann_begin_date = (datetime.datetime.strptime(react_ann_end_date,
                    "%Y-%m-%d").date() - datetime.timedelta(days=self.Parameters.birzha_ann_days_num)).strftime('%Y-%m-%d')

                logging.info('Preparing birzha annotations...')
                task = YdoRunYQL2(
                    self, 
                    yql_script='prepare_reactions_annotations.yql',
                    custom_placeholders={
                        '%RES_PATH%': temp_reactions_table,
                        '%BEGIN_DATE%': react_ann_begin_date,
                        '%END_DATE%': react_ann_end_date 
                    }
                )
                task.save()
                task.enqueue()
                self.Context.subtask_id = task.id
                raise sdk2.WaitTask(task, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK], wait_all=True)

        sessions_table = client.create_temp_table()
        if self.Parameters.end_date:
            end_date = self.Parameters.end_date
        else:
            end_date = (datetime.datetime.now() - datetime.timedelta(days=3)).strftime('%Y%m%d')

        sessions_cmd = [
            str(sdk2.ResourceData(self.Parameters.sessions_preparer).path),
            '-e', end_date,
            '-d', str(self.Parameters.days_num),
            '--session-info', sessions_table
        ]

        if self.Parameters.use_profile_clicks:
            sessions_cmd.append('--use-profiles')

        if self.Parameters.no_promo:
            sessions_cmd.append('--no-promo')

        if self.Parameters.cut_stopwords:
            sessions_cmd.append('--cut-stopwords')

        logging.info('Preparing sessions...')
        execute_cmd(sessions_cmd, 'ydo_ann_indexer', 'Failed to prepare sessions', env=env)

        stat_cmd = [
            str(sdk2.ResourceData(self.Parameters.stat_calculator).path),
            '--session-info', sessions_table,
            '--out', self.Parameters.out_table,
        ]

        if self.Parameters.external_ann:
            stat_cmd.extend(['--external-ann', self.Parameters.external_ann])

        if self.Parameters.quorum_ann:
            stat_cmd.append('--quorum-ann')

        if self.Parameters.proto_out:
            stat_cmd.append('--proto-out')

        if self.Parameters.quorum_mode:
            stat_cmd.extend(['--quorum-mode', self.Parameters.quorum_mode])

        if self.Parameters.real_region:
            stat_cmd.append('--real-region')

        if self.Parameters.end_date:
            end_date = self.Parameters.end_date
        else:
            end_date = (datetime.datetime.now() - datetime.timedelta(days=3)).strftime('%Y%m%d')

        if self.Parameters.calc_birzha_quorum_ann:
            stat_cmd.append('--add-birzha-quorum-ann')

        if self.Parameters.calc_birzha_ann:
            stat_cmd.extend(['--reaction-info', temp_reactions_table])

        logging.info('Calculating statistics...')
        execute_cmd(stat_cmd, 'ydo_ann_indexer', 'Failed to calc stat', env=env)
        
        if self.Parameters.calc_birzha_ann:
            client.remove(temp_reactions_table)
