# -*- coding: utf-8 -*-
import logging
import os
import tempfile

import sandbox.common.types.task as ctt
from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    get_now_utc,
    execute_cmd,
    YdoFeaturesJson,
)

from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump

from sandbox.projects.ydo.ferryman import (
    check_row_count,
    generate_ferryman_request_params,
    get_full_request_url,
    send_tables_to_ferryman,
    wait_ferryman,
)

from sandbox.projects.common.geosearch.utils import unpack_file
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables

from sandbox.projects.ydo.resource_types import (
    YdoDssmModel,
    YdoOrderReactionPredictionRankingDssmModel,
    YdoPruningModel,
    YdoReformulationsDssmModel,
    YdoSearchConnectionsRankingDssmModel,
    YdoSendToSaasLbExecutable,
    YdoWorkerOrderMatchingDssmModel,
)

from sandbox.sandboxsdk import environments


ProfileStateValues = {'_published': 0, '_unpublished': 1, '_not_correct': 2}


# Resource types
class YdoIndexerExecutable(sdk2.Resource):
    """
        Generate table with proto documents for SAAS
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoIndexer(sdk2.Task):
    '''Task that creates SAAS index and uploads it'''

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("SAAS specific parameters") as saas_block:
            saas_namespace = sdk2.parameters.String('SAAS namespace (kps)', required=True)

            service_name = sdk2.parameters.String('Service name, e.g. ydo_search, ydo_priemka, beware the underscore (using for factor whitelist and ferryman api url constructing')

            self_tvm_id = sdk2.parameters.Integer('Self tvm id', required=False)
            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=False)

            with sdk2.parameters.RadioGroup('SAAS ctype name') as ctype:
                ctype.values[''] = ctype.Value('')
                ctype.values['stable'] = ctype.Value('stable')
                ctype.values['stable_ydo'] = ctype.Value('stable_ydo')
                ctype.values['prestable_ydo'] = ctype.Value('prestable_ydo')
                ctype.values['stable_hamster'] = ctype.Value('stable_hamster')
                ctype.values['stable_hamster_ydo'] = ctype.Value('stable_hamster_ydo')

            use_ferryman = sdk2.parameters.Bool('Use ferryman delivery', default=True)
            with use_ferryman.value[True]:
                ferryman_ranking_name = sdk2.parameters.String('Ferryman name, use if you really know what you doing', required=False)
                ferryman_kv_name = sdk2.parameters.String('Ferryman kv name, use if you really know what you doing', required=False)

            use_saas_lb = sdk2.parameters.Bool('Use saas logbroker for sending documents', default=False)
            with use_saas_lb.value[True]:
                send_to_saas_lb = sdk2.parameters.Resource(
                    'send to saas lb executable',
                    resource_type=YdoSendToSaasLbExecutable,
                    required=False
                )

                kv_service = sdk2.parameters.String('Service name for kv saas lb', required=False)
                search_service = sdk2.parameters.String('Service name for search saas lb', required=False)
                ctype_saas_lb = sdk2.parameters.String('Ctype saas lb', required=False)
                logbroker_host = sdk2.parameters.String('Logbroker host', required=False, default="logbroker.yandex.net")
                logbroker_tvm_id = sdk2.parameters.Integer('Destination logbroker tvm id', required=False)
                document_ttl_in_days = sdk2.parameters.Integer("After how many days documents will be deleted", required=False)

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            indexer = sdk2.parameters.Resource(
                'indexer executable',
                resource_type=YdoIndexerExecutable,
                required=True
            )

            geobase_snapshot = sdk2.parameters.Resource(
                'Geobase 6 snapshot',
                resource_type=GEODATA6BIN_STABLE,
                required=True
            )

            geodata_tz_snapshot = sdk2.parameters.Resource(
                'Geodata tz',
                resource_type=GEODATATZDATA_STABLE,
                required=True
            )

            features_json = sdk2.parameters.Resource(
                'Features',
                resource_type=YdoFeaturesJson,
                required=True
            )

            rubrics_json = sdk2.parameters.Resource(
                'merged rubrics.json',
                resource_type=YdoMergedRubricsSmallDump,
                required=True
            )

        with sdk2.parameters.Group("Model resources") as model_block:
            dssm_model = sdk2.parameters.Resource(
                'DSSM model',
                resource_type=YdoDssmModel,
                required=False
            )

            reformulations_dssm_model = sdk2.parameters.Resource(
                'reformulations DSSM model',
                resource_type=YdoReformulationsDssmModel,
                required=False
            )

            worker_order_matching_dssm_model = sdk2.parameters.Resource(
                'worker order matching DSSM model',
                resource_type=YdoWorkerOrderMatchingDssmModel,
                required=False
            )

            order_reaction_prediction_ranking_dssm_model = sdk2.parameters.Resource(
                'order reaction prediction ranking DSSM model',
                resource_type=YdoOrderReactionPredictionRankingDssmModel,
                required=False
            )

            search_connections_ranking_dssm_model = sdk2.parameters.Resource(
                'search connections ranking DSSM model',
                resource_type=YdoSearchConnectionsRankingDssmModel,
                required=False
            )

            pruning_model = sdk2.parameters.Resource(
                'pruning model',
                resource_type=YdoPruningModel,
                required=False
            )

        with sdk2.parameters.Group("Output parameters") as output_block:
            res_dir = sdk2.parameters.String('Index resulting directory', required=True)

        with sdk2.parameters.Group("Common parameters") as common_block:
            yt_backup_dir = sdk2.parameters.String('Path to database backup on YT', required=True)

            factors_whitelist = sdk2.parameters.Bool('Use factor whitelist from backend', default=True)

            additional_factors = sdk2.parameters.String('Path to table with additional factors', required=False)

            additional_factors_by_card = sdk2.parameters.String('Path to table with additional factors by card', required=False)

            annotations = sdk2.parameters.String('Path to table with annotations', required=False)

            workers_ctr_factors = sdk2.parameters.String('Path to table with worker to ctr factors', required=False)

            rubric_click_factors = sdk2.parameters.String('Path to table with rubric click factors', required=False)

            price_stats_path = sdk2.parameters.String('Path to table with average price stats', required=False)

            ogrn_whitelist = sdk2.parameters.String('Path to table with OGRNs whitelist (allow doubles)', required=False)

        with sdk2.parameters.Group("Vertical specific parameters") as vertical_specific_block:
            with sdk2.parameters.String('Vertical', multiline=True, required=True) as vertical:
                vertical.values.ydo = vertical.Value('ydo')
                vertical.values.ydo_experiment = vertical.Value('ydo_experiment')
                vertical.values.sprav = vertical.Value('sprav')
                vertical.values.docdoc = vertical.Value('docdoc')

            with vertical.value['sprav']:
                sprav_dir = sdk2.parameters.String('Path to sprav directory', required=False)

                sprav_to_workers_path = sdk2.parameters.String('Path to table with sprav_to_workers table', required=False)

                clear_online_orgs_addresses = sdk2.parameters.Bool('Clear addresses and areas if org has online tag', default=False)

            with vertical.value['ydo'], vertical.value['ydo_experiment']:
                with sdk2.parameters.Group("Altay integration parameters") as altay_block:
                    update_addresses_with_sprav = sdk2.parameters.Bool('Change worker\'s address to org or chain address', default=False)

                    workers_to_sprav_companies_result_path = sdk2.parameters.String('Path to result of workers_to_sprav_companies querry', required=False)

                    altay_company_to_chain_path = sdk2.parameters.String('Path to sprav company_to_chain table', required=False)

                    filter_far_areas_sprav_linked = sdk2.parameters.Bool('Filter areas far from sprav linked addresses', default=False)

                    far_areas_max_distance = sdk2.parameters.Integer('Max distance from addresses for areas of sprav linked workers', default=200000)

                    cut_areas_sprav_not_linked = sdk2.parameters.Bool('Cut areas of orgs not linked with sprav', default=False)

                    areas_sprav_not_linked_max_distance = sdk2.parameters.Integer('Max distance between areas of not sprav linked orgs', default=200000)

                    clear_linked_sprav_online_orgs_addresses = sdk2.parameters.Bool('Remove addresses, areas and main_address from workers linked with sprav online orgs', default=False)

                with sdk2.parameters.Group("Yabs and promo integration parameters") as yabs_block:
                    certificates_path = sdk2.parameters.String('Path to table with direct certificates info', required=False)

                    yabs_banners_table = sdk2.parameters.String('Path to table with yabs banners', required=False)

                    marketing_attracted_workers_table = sdk2.parameters.String('Path to table with information about workers attracted with advertising', required=False)

                    promo_trial_table = sdk2.parameters.String('Path to table with workers on trial promotion', required=False)

                    minus_phrases_table = sdk2.parameters.String('Path to table with minus phrases for promo workers', required=False)

                with sdk2.parameters.Group("Other parameters") as other_block:
                    geometry_path = sdk2.parameters.String('Path to table with geometries', required=False)

                    worker_youtube_videos_table = sdk2.parameters.String('Path to worker youtube videos', required=False)

                    regular_coords_table = sdk2.parameters.String('Path to table with workers\' regular coordinates', required=False)

                    antifraud_dublicates = sdk2.parameters.String('Path to table with antifraud dublicates', required=False)

                    orders_factors_data = sdk2.parameters.String('Path to table with factors about activity in orders', required=False)

                    rubric_to_orders_factors_data = sdk2.parameters.String('Path to table with factors about activity in orders by rubric', required=False)

                    verified_reviews_table = sdk2.parameters.String('Path to verified reviews table', required=False)

                    next_version_verified_reviews_table = sdk2.parameters.String('Path to next version verified reviews table', required=False)

                    pruning_polynom = sdk2.parameters.String('Polynom to be used for pruning, overrides model', required=False)

        with sdk2.parameters.Group("Miscellaneous parameters") as miscellaneous_block:
            additional_cmd_options = sdk2.parameters.String('Additional command line options')

            yt_log_level = sdk2.parameters.String('Yt log level', required=False)

            enable_monitoring = sdk2.parameters.Bool('Enable pushing statistics to solomon', default=False)

            with enable_monitoring.value[True]:
                solomon_label = sdk2.parameters.String('Label in solomon for signal dispatching')

            degrade_level = sdk2.parameters.Float('Max allowed difference between tables row counts', default=1.0)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        cores = 1
        disk_space = 60 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def create_index(self):
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['YT_PROXY'] = '{}.yt.yandex.net'.format(self.yt_host)
        env['YT_LOG_LEVEL'] = self.Parameters.yt_log_level if self.Parameters.yt_log_level else 'INFO'
        if self.Parameters.enable_monitoring:
            env['SOLOMON_TOKEN'] = sdk2.Vault.data(self.owner, 'solomon-token')

        unpack_file(str(sdk2.ResourceData(self.Parameters.geodata_tz_snapshot).path), str(sdk2.Path.cwd()))

        self.Context.index_ts = get_now_utc()

        def table_name(template):
            return os.path.join(self.Parameters.res_dir, template.format(ts=self.Context.index_ts, kps=self.Parameters.saas_namespace))

        self.Context.out_ranking_table = table_name('{ts}_ranking')
        self.Context.out_kv_table = table_name('{ts}_kv')
        self.Context.out_ids_table = table_name('{ts}_ids_kps={kps}')

        self.Context.index_res_local_path = os.path.join(str(sdk2.Path.cwd()), 'index')
        self.Context.kv_res_local_path = os.path.join(str(sdk2.Path.cwd()), 'kv')

        logging.info('Indexing...')

        if self.Parameters.self_tvm_id and self.Parameters.tvm_secret:
            env['TVM_CLIENT_ID'] = str(self.Parameters.self_tvm_id)
            env['TVM_SECRET'] = self.Parameters.tvm_secret.data()['TVM_SECRET']

        execute_cmd(
            [
                str(sdk2.ResourceData(self.Parameters.indexer).path),
                '--dump_from_yt', self.Parameters.yt_backup_dir,
                '--write_to_yt',
                '--write_proxy', '{}.yt.yandex.net'.format(self.yt_host),
                '--index_res_path', self.Context.out_ranking_table,
                '--kv_res_path', self.Context.out_kv_table,
                '--ids_res_path', self.Context.out_ids_table,
                '--geobase_file', str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path),
                '--geodata_tz_location', str(sdk2.Path.cwd().joinpath('zones_bin')),
                '--features_json_path', str(sdk2.ResourceData(self.Parameters.features_json).path),
            ]
            + (
                [
                    '--index_res_local_path', self.Context.index_res_local_path,
                    '--kv_res_local_path', self.Context.kv_res_local_path
                ] if self.Parameters.use_saas_lb else []
            )
            + (['--rubrics_json', str(sdk2.ResourceData(self.Parameters.rubrics_json).path)] if self.Parameters.rubrics_json else [])
            + (self.Parameters.additional_cmd_options.split() if self.Parameters.additional_cmd_options else [])
            + (
                [
                    '--service_name', self.Parameters.service_name,
                    '--saas_ctype', self.Parameters.ctype,
                ] if self.Parameters.factors_whitelist else []
            )
            + (['--additional_factors', self.Parameters.additional_factors] if self.Parameters.additional_factors else [])
            + (['--additional_factors_by_card', self.Parameters.additional_factors_by_card] if self.Parameters.additional_factors_by_card else [])
            + (['--annotations_table', self.Parameters.annotations] if self.Parameters.annotations else [])
            + (['--workers_ctr_factors', self.Parameters.workers_ctr_factors] if self.Parameters.workers_ctr_factors else [])
            + (['--rubric_click_factors', self.Parameters.rubric_click_factors] if self.Parameters.rubric_click_factors else [])
            + (['--certs_path', self.Parameters.certificates_path] if self.Parameters.certificates_path else [])
            + (['--sprav_dir', self.Parameters.sprav_dir] if self.Parameters.sprav_dir else [])
            + (['--sprav_to_workers_path', self.Parameters.sprav_to_workers_path] if self.Parameters.sprav_to_workers_path else [])
            + (['--clear_online_orgs_addresses'] if self.Parameters.clear_online_orgs_addresses else [])
            + (['--workers_to_geom_path', self.Parameters.geometry_path] if self.Parameters.geometry_path else [])
            + (['--price_stats_table', self.Parameters.price_stats_path] if self.Parameters.price_stats_path else [])
            + ['--vertical', self.Parameters.vertical]
            + (['--enable-monitoring'] if self.Parameters.enable_monitoring else [])
            + (['--solomon-label', self.Parameters.solomon_label] if self.Parameters.solomon_label else [])
            + (['--worker_youtube_videos_table', self.Parameters.worker_youtube_videos_table] if self.Parameters.worker_youtube_videos_table else [])
            + (['--yabs_banners_table', self.Parameters.yabs_banners_table] if self.Parameters.yabs_banners_table else [])
            + (['--promo_trial_table', self.Parameters.promo_trial_table] if self.Parameters.promo_trial_table else [])
            + (['--regular_coords_table', self.Parameters.regular_coords_table] if self.Parameters.regular_coords_table else [])
            + (['--minus_phrases_table', self.Parameters.minus_phrases_table] if self.Parameters.minus_phrases_table else [])
            + (['--dssm_model', str(sdk2.ResourceData(self.Parameters.dssm_model).path)] if self.Parameters.dssm_model else [])
            + (['--reformulations_dssm_model', str(sdk2.ResourceData(self.Parameters.reformulations_dssm_model).path)] if self.Parameters.reformulations_dssm_model else [])
            + (['--worker_order_matching_dssm_model', str(sdk2.ResourceData(self.Parameters.worker_order_matching_dssm_model).path)] if self.Parameters.worker_order_matching_dssm_model else [])
            + (
                [
                    '--order_reaction_prediction_ranking_dssm_model', str(sdk2.ResourceData(self.Parameters.order_reaction_prediction_ranking_dssm_model).path),
                ] if self.Parameters.order_reaction_prediction_ranking_dssm_model else [])
            + (
                [
                    '--search_connections_dssm_model', str(sdk2.ResourceData(self.Parameters.search_connections_ranking_dssm_model).path),
                ] if self.Parameters.search_connections_ranking_dssm_model else [])
            + (['--pruning_polynom', self.Parameters.pruning_polynom] if self.Parameters.pruning_polynom else [])
            + (['--pruning_mx_model', str(sdk2.ResourceData(self.Parameters.pruning_model).path)] if self.Parameters.pruning_model else [])
            + (
                [
                    '--workers_to_sprav_companies_result_path', self.Parameters.workers_to_sprav_companies_result_path,
                    '--altay_company_to_chain_path', self.Parameters.altay_company_to_chain_path,
                ] if self.Parameters.update_addresses_with_sprav else []
            )
            + (['--marketing_attracted_workers_table', self.Parameters.marketing_attracted_workers_table] if self.Parameters.marketing_attracted_workers_table else [])
            + (['--filter_far_areas_sprav_linked', '--far_areas_max_distance', str(self.Parameters.far_areas_max_distance)] if self.Parameters.filter_far_areas_sprav_linked else [])
            + (
                [
                    '--cut_areas_sprav_not_linked',
                    '--areas_sprav_not_linked_max_distance',
                    str(self.Parameters.areas_sprav_not_linked_max_distance)
                ] if self.Parameters.cut_areas_sprav_not_linked else []
            )
            + (['--clear_linked_sprav_online_orgs_addresses'] if self.Parameters.clear_linked_sprav_online_orgs_addresses else [])
            + (['--antifraud_dublicates', self.Parameters.antifraud_dublicates] if self.Parameters.antifraud_dublicates else [])
            + (['--orders_factors_data', self.Parameters.orders_factors_data] if self.Parameters.orders_factors_data else [])
            + (['--rubric_to_orders_factors_data', self.Parameters.rubric_to_orders_factors_data] if self.Parameters.rubric_to_orders_factors_data else [])
            + (['--verified_reviews_path', self.Parameters.verified_reviews_table] if self.Parameters.verified_reviews_table else [])
            + (['--next_version_verified_reviews_path', self.Parameters.next_version_verified_reviews_table] if self.Parameters.next_version_verified_reviews_table else [])
            + (['--ogrn_whitelist', self.Parameters.ogrn_whitelist] if self.Parameters.ogrn_whitelist else [])
            + (['--self_tvm_id', 'TVM_CLIENT_ID', '--tvm_secret', 'TVM_SECRET'] if self.Parameters.self_tvm_id and self.Parameters.tvm_secret else []),
            'ydo_indexer',
            'Failed to create index',
            env=env
        )

    def link(self):
        tables_to_link = {
            self.Context.out_ids_table: os.path.join(self.Parameters.res_dir, 'current_ids'),
        }

        for suffix in ProfileStateValues.keys():
            tables_to_link.update({
                (self.Context.out_ranking_table + suffix): os.path.join(self.Parameters.res_dir, ('current_index' + suffix)),
                (self.Context.out_kv_table + suffix): os.path.join(self.Parameters.res_dir, ('current_kv' + suffix)),
            })

        self.Context.num_tables = len(tables_to_link)

        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.yt_host,
            yt_vault_token='yt-token',
            yt_tables=tables_to_link
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    # текущая философия состоит из нескольких правил:
    # 0. ctype'ы stable_hamster и stable в saas должны смотреть на один ferryman (т.е. никогда не отличаться)
    # 1. имя ранжирующего сааса и kv связаны как kv_name = ranking_name + '_workers' (кроме прода, там ydo_search/ydo_workers)
    # 2. в случае ctype prestable к именам ferryman добавляется суффикс '-prestable' (чтобы на тестовый контур можно было катать отдельно)
    def calc_ferryman_ranking_name(self):
        service_name = self.Parameters.service_name.replace('_', '-')

        if self.Parameters.ctype == 'prestable_ydo':
            return '-'.join((service_name, 'prestable'))
        else:
            return service_name

    def calc_ferryman_kv_name(self):
        service_name = self.Parameters.service_name.replace('_', '-')

        if service_name == 'ydo-search':
            kv_service_name = 'ydo-workers'
        else:
            kv_service_name = service_name + '-workers'

        if self.Parameters.ctype == 'prestable_ydo':
            return '-'.join((kv_service_name, 'prestable'))
        else:
            return kv_service_name

    def send_with_ferryman(self):
        with self.memoize_stage.sending:
            ferryman_ranking_name = self.calc_ferryman_ranking_name()
            ferryman_kv_name = self.calc_ferryman_kv_name()

            def get_table_mapping(out_table, saas_namespace):
                out_tables2saas_namespaces = {}
                if self.Parameters.vertical == 'ydo':
                    for suffix, shift in ProfileStateValues.items():
                        out_tables2saas_namespaces.update({(out_table + suffix): str(int(saas_namespace) - shift)})
                else:
                    out_tables2saas_namespaces.update({(out_table + '_published'): saas_namespace})
                return out_tables2saas_namespaces

            if self.Parameters.ferryman_ranking_name:
                self.Context.ferryman_ranking_name = self.Parameters.ferryman_ranking_name
            else:
                self.Context.ferryman_ranking_name = ferryman_ranking_name

            if self.Parameters.ferryman_kv_name:
                self.Context.ferryman_kv_name = self.Parameters.ferryman_kv_name
            else:
                self.Context.ferryman_kv_name = ferryman_kv_name

            ferryman_ranking_request = generate_ferryman_request_params(
                get_table_mapping(
                    self.Context.out_ranking_table,
                    self.Parameters.saas_namespace
                ),
                self.Context.index_ts,
                cluster=self.yt_host,
                proto=True
            )

            ferryman_kv_request = generate_ferryman_request_params(
                get_table_mapping(
                    self.Context.out_kv_table,
                    self.Parameters.saas_namespace
                ),
                self.Context.index_ts,
                cluster=self.yt_host,
                proto=True
            )

            self.Context.ranking_url = get_full_request_url(
                ferryman_ranking_request,
                self.Context.ferryman_ranking_name
            )

            self.Context.kv_url = get_full_request_url(
                ferryman_kv_request,
                self.Context.ferryman_kv_name
            )

            check_row_count(
                ferryman_ranking_request,
                self.Context.ferryman_ranking_name,
                sdk2.Vault.data(self.owner, 'yt-token'),
                self.yt_host,
                self.Parameters.degrade_level
            )

            check_row_count(
                ferryman_kv_request,
                self.Context.ferryman_kv_name,
                sdk2.Vault.data(self.owner, 'yt-token'),
                self.yt_host,
                self.Parameters.degrade_level
            )

            self.Context.ferryman_ranking_batch_id = send_tables_to_ferryman(
                self.Context.ferryman_ranking_name,
                ferryman_ranking_request
            )

            self.Context.ferryman_kv_batch_id = send_tables_to_ferryman(
                self.Context.ferryman_kv_name,
                ferryman_kv_request
            )

    def wait_ferryman(self):
        with self.memoize_stage.wait_ferryman(9):
            wait_ferryman(self.Context.ferryman_ranking_name, self.Context.ferryman_ranking_batch_id, time_to_wait=20 * 60)
            wait_ferryman(self.Context.ferryman_kv_name, self.Context.ferryman_kv_batch_id, time_to_wait=10 * 60)

    def send_with_saas_lb(self):
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['TVM_SECRET'] = self.Parameters.tvm_secret.data()['TVM_SECRET']

        cmd = [
            str(sdk2.ResourceData(self.Parameters.send_to_saas_lb).path),
            "--kv_service", self.Parameters.kv_service,
            "--search_service", self.Parameters.search_service,
            "--ctype_saas_lb", self.Parameters.ctype_saas_lb,
            "--logbroker_host", self.Parameters.logbroker_host,
            "--self_tvm_id", str(self.Parameters.self_tvm_id),
            "--logbroker_tvm_id", str(self.Parameters.logbroker_tvm_id),
            "--kps", self.Parameters.saas_namespace,
            "--read_proxy", '{}.yt.yandex.net'.format(self.yt_host),
            "--kv_docs_path", self.Context.kv_res_local_path + "_published",
            "--search_docs_path", self.Context.index_res_local_path + "_published",
            "--timestamp", str(self.Context.index_ts),
            "--document_ttl_in_days", str(self.Parameters.document_ttl_in_days),
        ]

        execute_cmd(cmd, 'ydo_send_to_saas_lb', 'Failed to update index', env=env)

    def on_execute(self):
        self.yt_host = 'hahn'

        with self.memoize_stage.firststage:
            if not (self.Parameters.ctype and self.Parameters.service_name):
                raise RuntimeError('You should set both ctype and service name')

            self.create_index()

        if self.Parameters.use_ferryman:
            self.send_with_ferryman()

        if self.Parameters.use_saas_lb:
            with self.memoize_stage.sending_to_lb:
                self.send_with_saas_lb()

        if self.Parameters.use_ferryman:
            self.wait_ferryman()

        with self.memoize_stage.link_to_current:
            self.link()

        with self.memoize_stage.clean_folder:
            clean_folder(self, self.Parameters.res_dir, yt_host=self.yt_host, history_size=4 * self.Context.num_tables)

        logging.info('Done')
