import logging
import os

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo import (
    execute_cmd,
    get_now_utc,
    ydo_releasers,
)
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables
from sandbox.projects.ydo.ferryman import (
    start_ferryman_process,
    wait_ferryman,
)


class YdoSnippetsIndexerExecutable(sdk2.Resource):
    """
        Generate table with proto documents for SAAS
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoSnippetsIndexer(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        indexer = sdk2.parameters.Resource(
            'snippets indexer executable',
            resource_type=YdoSnippetsIndexerExecutable,
            required=True
        )

        ferryman_kv_name = sdk2.parameters.String('Ferryman kv name', required=True)

        worker_saas_namespace = sdk2.parameters.String('SAAS namespace (kps) for worker', required=True)
        card_saas_namespace = sdk2.parameters.String('SAAS namespace (kps) for card', required=True)
        category_saas_namespace = sdk2.parameters.String('SAAS namespace (kps) for category', required=True)
        org_permalink_saas_namespace = sdk2.parameters.String('SAAS namespace (kps) for org_permalink', required=True)
        chain_permalink_puid_saas_namespace = sdk2.parameters.String('SAAS namespace (kps) for chain_permalink:puid', required=True)

        degrade_level = sdk2.parameters.Float('Max allowed difference between tables row counts', default=1.0)

        res_dir = sdk2.parameters.String('Index resulting directory', required=True)

        additional_factors = sdk2.parameters.String('Path to table with additional factors', required=False)
        additional_factors_by_card = sdk2.parameters.String('Path to table with additional factors by card', required=False)
        altay_company_to_chain_path = sdk2.parameters.String('Path to sprav company_to_chain table', required=False)
        annotations_table = sdk2.parameters.String('Path to table with annotations table', required=False)
        antifraud_dublicates = sdk2.parameters.String('Path to table with antifraud dublicates', required=False)
        certs_path = sdk2.parameters.String('Path to table with certs path', required=False)
        marketing_attracted_workers_table = sdk2.parameters.String('Path to table with marketing attracted workers table', required=False)
        minus_phrases_table = sdk2.parameters.String('Path to table with minus phrases for promo workers', required=False)
        next_version_verified_reviews_path = sdk2.parameters.String('Path to next version verified reviews table', required=False)
        orders_factors_data = sdk2.parameters.String('Path to table with orders factors data', required=False)
        promo_trial_table = sdk2.parameters.String('Path to table with promo trial table', required=False)
        puid_to_worker_id_table = sdk2.parameters.String('Path to puid_to_worker_id table', required=False)
        regular_coords_table = sdk2.parameters.String('Path to table with regular coords table', required=False)
        rubric_click_factors = sdk2.parameters.String('Path to table with rubric click factors', required=False)
        rubric_to_orders_factors_data = sdk2.parameters.String('Path to table with factors about activity in orders by rubric', required=False)
        verified_reviews_path = sdk2.parameters.String('Path to verified reviews table', required=False)
        worker_youtube_videos_table = sdk2.parameters.String('Path to table with worker youtube videos table', required=False)
        workers_ctr_factors = sdk2.parameters.String('Path to table with workers ctr factors', required=False)
        workers_to_geom_path = sdk2.parameters.String('Path to table with workers to geom path', required=False)
        workers_to_sprav_companies_result_path = sdk2.parameters.String('Path to result of workers_to_sprav_companies querry', required=False)
        yabs_banners_table = sdk2.parameters.String('Path to table with yabs banners table', required=False)

    def create_index(self):
        self.Context.timestamp = get_now_utc()

        def table_name(template, kps):
            return os.path.join(self.Parameters.res_dir, template.format(ts=self.Context.timestamp, kps=kps))

        self.Context.out_worker_table = table_name('{ts}_worker_kps={kps}', self.Parameters.worker_saas_namespace)
        self.Context.out_card_table = table_name('{ts}_card_kps={kps}', self.Parameters.card_saas_namespace)
        self.Context.out_category_table = table_name('{ts}_category_kps={kps}', self.Parameters.category_saas_namespace)
        self.Context.out_org_permalink_table = table_name('{ts}_org_permalink_kps={kps}', self.Parameters.org_permalink_saas_namespace)
        self.Context.out_chain_permalink_puid_table = table_name('{ts}_chain_permalink_puid_kps={kps}', self.Parameters.chain_permalink_puid_saas_namespace)

        logging.info('Indexing...')

        cmd = [
            str(sdk2.ResourceData(self.Parameters.indexer).path),
            "--token", sdk2.Vault.data(self.owner, 'yt-token'),
            "--worker_res_table", self.Context.out_worker_table,
            "--card_res_table", self.Context.out_card_table,
            "--category_res_table", self.Context.out_category_table,
            "--org_permalink_res_table", self.Context.out_org_permalink_table,
            "--chain_permalink_puid_res_table", self.Context.out_chain_permalink_puid_table,
        ]

        tables = (
            "additional_factors",
            "additional_factors_by_card",
            "altay_company_to_chain_path",
            "annotations_table",
            "antifraud_dublicates",
            "certs_path",
            "marketing_attracted_workers_table",
            "minus_phrases_table",
            "next_version_verified_reviews_path",
            "orders_factors_data",
            "promo_trial_table",
            "puid_to_worker_id_table",
            "regular_coords_table",
            "rubric_click_factors",
            "rubric_to_orders_factors_data",
            "verified_reviews_path",
            "worker_youtube_videos_table",
            "workers_ctr_factors",
            "workers_to_geom_path",
            "workers_to_sprav_companies_result_path",
            "yabs_banners_table",
        )
        for name in tables:
            path = getattr(self.Parameters, name)
            if path:
                cmd += [
                    "--" + name,
                    path,
                ]

        execute_cmd(cmd, 'ydo_snippets_indexer', 'Failed to create snippets index')

    def link(self):
        tables_to_link = {
            self.Context.out_worker_table: os.path.join(self.Parameters.res_dir, 'current_worker_kv'),
            self.Context.out_card_table: os.path.join(self.Parameters.res_dir, 'current_card_kv'),
            self.Context.out_category_table: os.path.join(self.Parameters.res_dir, 'current_category_kv'),
            self.Context.out_org_permalink_table: os.path.join(self.Parameters.res_dir, 'current_org_permalink_kv'),
            self.Context.out_chain_permalink_puid_table: os.path.join(self.Parameters.res_dir, 'current_chain_permalink_puid_kv'),
        }
        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.yt_host,
            yt_vault_token='yt-token',
            yt_tables=tables_to_link,
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def on_execute(self):
        self.yt_host = 'hahn'

        with self.memoize_stage.firststage:
            self.create_index()

            ferryman_args = {
                "ferryman_name": self.Parameters.ferryman_kv_name,
                "timestamp": self.Context.timestamp,
                "yt_token": sdk2.Vault.data(self.owner, 'yt-token'),
                "degrade_level": self.Parameters.degrade_level,
                "cluster": self.yt_host,
                "proto": True,
            }
            out_tables2saas_namespaces = {
                self.Context.out_worker_table: self.Parameters.worker_saas_namespace,
                self.Context.out_card_table: self.Parameters.card_saas_namespace,
                self.Context.out_category_table: self.Parameters.category_saas_namespace,
                self.Context.out_org_permalink_table: self.Parameters.org_permalink_saas_namespace,
                self.Context.out_chain_permalink_puid_table: self.Parameters.chain_permalink_puid_saas_namespace,
            }
            self.Context.ferryman_batch_id = start_ferryman_process(out_tables2saas_namespaces, **ferryman_args)

        with self.memoize_stage.wait_ferryman(9):
            wait_ferryman(self.Parameters.ferryman_kv_name, self.Context.ferryman_batch_id)

        with self.memoize_stage.link_to_current:
            self.link()

        with self.memoize_stage.clean_folder:
            clean_folder(self, self.Parameters.res_dir, yt_host=self.yt_host, history_size=18)

        logging.info('Done')
