# -*- coding: utf-8 -*-
import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    YdoFeaturesJson,
)
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.ydo.resource_types import (
    YdoTestMarketExecutable,
    YdoDispatchShooterConfigs,
    YdoServiceModelCategoriesJson,
)
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump


class YdoTestMarketPackages(sdk2.Task):
    """Task shoots with market packages at busy_periods"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            executable = sdk2.parameters.Resource(
                "test market executable",
                resource_type=YdoTestMarketExecutable,
                required=True,
            )
            config = sdk2.parameters.Resource(
                "dispatch shooter config",
                resource_type=YdoDispatchShooterConfigs,
                required=True,
            )
            geobase_snapshot = sdk2.parameters.Resource(
                'Geobase 6 snapshot',
                resource_type=GEODATA6BIN_STABLE,
                required=True,
            )
            rubrics_json = sdk2.parameters.Resource(
                'merged rubrics.json',
                resource_type=YdoMergedRubricsSmallDump,
                required=True,
            )
            features_json = sdk2.parameters.Resource(
                'Features',
                resource_type=YdoFeaturesJson,
                required=True,
            )
            service_model_categories_json = sdk2.parameters.Resource(
                'service model categories json',
                resource_type=YdoServiceModelCategoriesJson,
                required=True,
            )

        with sdk2.parameters.Group("Environment") as env_block:
            with sdk2.parameters.RadioGroup('Ydo ctype name') as ctype:
                ctype.values['hamster'] = ctype.Value('hamster')
                ctype.values['testing'] = ctype.Value('testing')

            ydb_token = sdk2.parameters.YavSecret("Ydb token in yav", required=True)
            tvm_secret = sdk2.parameters.YavSecret("Tvm secret in yav", required=True)

        with sdk2.parameters.Group("Input parameters") as input_params:
            external_services = sdk2.parameters.String("YT external services table", required=True)
            result_table = sdk2.parameters.String("YT results table", required=True)

    def run(self):
        self.Context.yt_vault_token = "yt-token"
        yt_token = sdk2.Vault.data(self.owner, self.Context.yt_vault_token)

        env = os.environ.copy()
        env["YT_TOKEN"] = yt_token
        env["YT_PROXY"] = "{}.yt.yandex.net".format(self.yt_host)
        env["TVM_SECRET"] = self.Parameters.tvm_secret.data()["TVM_SECRET"]
        env["YDB_TOKEN"] = self.Parameters.ydb_token.data()["YDB_TOKEN"]

        tmp_dir = tempfile.mkdtemp()
        os.symlink(str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path), os.path.join(tmp_dir, "geodata6.bin"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.rubrics_json).path), os.path.join(tmp_dir, "rubrics.json"))
        os.symlink(str(sdk2.ResourceData(self.Parameters.features_json).path), os.path.join(tmp_dir, "features.json"))
        os.symlink(
            str(sdk2.ResourceData(self.Parameters.service_model_categories_json).path),
            os.path.join(tmp_dir, "service_model_categories.json"),
        )

        logging.info("Pow! Pow!...")
        cmd = [
            str(sdk2.ResourceData(self.Parameters.executable).path),
            "--config",
            os.path.join(str(sdk2.ResourceData(self.Parameters.config).path), str(self.Parameters.ctype) + ".yaml"),
            "--dispatch_shooter_config",
            os.path.join(str(sdk2.ResourceData(self.Parameters.config).path), str(self.Parameters.ctype) + ".yaml"),
            "--read_proxy",
            "{}.yt.yandex.net".format(self.yt_host),
            "--write_proxy",
            "{}.yt.yandex.net".format(self.yt_host),
            "--external_services",
            self.Parameters.external_services,
            "--results_table",
            self.Parameters.result_table,
        ]

        execute_cmd(
            cmd,
            "test_market",
            "Failed to shoot busy_periods",
            env=env,
            cwd=tmp_dir,
        )

    def on_execute(self):
        self.yt_host = "hahn"

        with self.memoize_stage.running:
            self.run()

        logging.info("Done")
