import tempfile

from sandbox import sdk2
from sandbox.projects.common import file_utils

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.default_multislot_task import YdoMutlihostTask
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo import ydo_releasers


class YdoMarketCreateOrdersExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoMarketCreateOrdersResultFile(sdk2.Resource):
    pass


class YdoMarketCreateOrders(YdoMutlihostTask):
    class Parameters(generate_view(
        market_create_orders_exe=YdoMarketCreateOrdersExecutable,
        check_owner=True,
        with_environ=True,
        default_secret_env_vars={
            "YDB_TOKEN": "ydb-token",
            "TVM_SECRET_PROD": "ydo_tvm_secret_prod",
            "TVM_SECRET_TEST": "ydo_tvm_secret_test",
        },
    )):
        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_host = sdk2.parameters.String("YDB host", required=True)
            ydb_port = sdk2.parameters.String("YDB port", required=True)
            ydb_database = sdk2.parameters.String("YDB database", required=True)
            ydb_home = sdk2.parameters.String("YDB home", required=True)

        with sdk2.parameters.Group("Market create parameters") as market_block:
            url = sdk2.parameters.String("YDO url", required=True)
            ya_service_ids = sdk2.parameters.String("Market yaServiceIds", required=True, multiline=True)
            date = sdk2.parameters.String("Order date DD-MM-YYYY", required=False)
            with sdk2.parameters.RadioGroup("City") as city:
                city.values["default"] = city.Value(value="Default", default=True)
                city.values["moscow"] = city.Value(value="Moscow")
                city.values["spb"] = city.Value(value="SPB")
            orders_creation_check_delay = sdk2.parameters.Integer("Creation check delay", required=False, default=60)
            orders_creation_check_count = sdk2.parameters.Integer("Creation check count", required=False, default=15)

    def on_execute(self):
        env = self.Parameters.get_environ()
        if self.Parameters.url == 'test':
            env['TVM_SECRET'] = env['TVM_SECRET_TEST']
        elif self.Parameters.url == 'prod':
            env['TVM_SECRET'] = env['TVM_SECRET_PROD']

        ya_service_ids_tempfile = tempfile.NamedTemporaryFile()
        file_utils.write_file(ya_service_ids_tempfile.name, self.Parameters.ya_service_ids)
        ya_service_ids_tempfile.seek(0)

        result_data = sdk2.ResourceData(YdoMarketCreateOrdersResultFile(self, "from task: {}".format(self.id), "result.json"))

        gps = None
        if self.Parameters.city == 'moscow':
            gps = '37.587093,55.733974'
        elif self.Parameters.city == 'spb':
            gps = '30.357496,59.931774'

        cmd = [
            self.Parameters.market_create_orders_exe_path,
            "--ydb_host",
            self.Parameters.ydb_host,
            "--ydb_port",
            self.Parameters.ydb_port,
            "--ydb_database",
            self.Parameters.ydb_database,
            "--ydb_home",
            self.Parameters.ydb_home,
            "--url",
            self.Parameters.url,
            "--date",
            self.Parameters.date,
            "--market_services_path",
            ya_service_ids_tempfile.name,
            "--orders_creation_check_delay",
            str(self.Parameters.orders_creation_check_delay),
            "--orders_creation_check_count",
            str(self.Parameters.orders_creation_check_count),
            "--results_file",
            str(result_data.path),
            "--fake_market_orders",
            "--auto_fake_market_orders",
            "--raise_on_fail",
        ] + (['--gps', gps] if gps else [])

        execute_cmd(cmd, "ydo_market_create_orders", "Create failed", env=env)
