import os

from sandbox.common.errors import TaskError
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog

from sandbox.projects.ydo import ydo_releasers

from sandbox.projects.resource_types import OTHER_RESOURCE
from sandbox.sandboxsdk import environments


class YdoMessengerGuidsGetter(sdk2.Resource):
    '''Tool for registering puids in messenger'''
    releasable = True
    releasers = ydo_releasers


class YdoPersonalInfoPatcher(sdk2.Resource):
    '''Tool for patching personal infos'''
    releasable = True
    releasers = ydo_releasers


class YdoRegisterPuidsInMessenger(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="ydb-token", required=True)
            ydb_host = sdk2.parameters.String("Host", required=True)
            ydb_port = sdk2.parameters.String("Port", required=True)
            ydb_database = sdk2.parameters.String("Database", required=True)
            ydb_home = sdk2.parameters.String("Home", required=True)

        backup_workers_yt_path = sdk2.parameters.String('Path to workers table', required=True)
        personal_info_patcher = sdk2.parameters.Resource(
            'PersonalInfoPatcher',
            resource_type=YdoPersonalInfoPatcher,
            required=True,
        )
        with sdk2.parameters.Group('Messenger parameters'):
            guids_getter = sdk2.parameters.Resource(
                'MessengerGuidsGetter',
                resource_type=YdoMessengerGuidsGetter,
                required=True,
            )
            client_tvm_id = sdk2.parameters.String('Client TVM id', required=True)
            client_tvm_vault = sdk2.parameters.String('Client TVM vault name', required=True)
            mssngr_tvm_id = sdk2.parameters.String('Messenger TVM id', required=True)
            mssngr_host = sdk2.parameters.String('Messenger host api', required=True)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),  # for temporary usage
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def execute_cmd(self, cmd, cmd_name, inp_file=None, out_file=None, err_file=None, with_yt=False, env=None):
        def open_or_default(inp, default, open_mode='r'):
            return open(inp, open_mode) if inp else default

        env = env or os.environ.copy()
        if with_yt:
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, 'yt-token')
            env['YT_PROXY'] = 'hahn.yt.yandex.net'
        with ProcessLog(self, logger=cmd_name) as pl:
            if sp.Popen(cmd, stdin=open_or_default(inp_file, None), stdout=open_or_default(out_file, pl.stdout, 'w'), stderr=open_or_default(err_file, pl.stderr, 'w'), env=env).wait():
                raise TaskError('{} failed'.format(cmd_name))
            for f in [out_file, err_file]:
                if f:
                    sdk2.ResourceData(OTHER_RESOURCE(self, f, f)).ready()

    def register_puids(self, cmd_name, inp_file, out_file, err_file):
        register_puids_cmd = [
            str(sdk2.ResourceData(self.Parameters.guids_getter).path),
            '--tvm_secret', sdk2.Vault.data(self.owner, self.Parameters.client_tvm_vault),
            '--self_tvm_id', self.Parameters.client_tvm_id,
            '--mssngr_tvm_id', self.Parameters.mssngr_tvm_id,
            '--mssngr_api_host', self.Parameters.mssngr_host,
        ]
        self.execute_cmd(register_puids_cmd, cmd_name, inp_file, out_file, err_file)

    def register_workers(self):
        read_workers_cmd = [
            'yt', 'read',
            '--format', '<columns=[puid;personal_info]>schemaful_dsv',
            self.Parameters.backup_workers_yt_path,
        ]

        self.execute_cmd(read_workers_cmd, 'read_workers', out_file='puid2personal_info', with_yt=True)
        self.execute_cmd(['grep', '-v', '_messenger_guid'], 'filter_registered', inp_file='puid2personal_info', out_file='filtered_puid2personal_info')
        self.execute_cmd(['cut', '-f1'], 'leave_puids', inp_file='filtered_puid2personal_info', out_file='filtered_puids')
        self.execute_cmd(['uniq'], 'uniq_puids', inp_file='filtered_puids', out_file='filtered_uniq_puids')

        self.register_puids('register_puids', inp_file='filtered_uniq_puids', out_file='puid2guid', err_file='register.errors')

        env = os.environ.copy()
        if self.Parameters.ydb_host != 'ydb-services.yandex.net':
            env['YDB_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.ydb_vault_token)
        patch_personal_infos = [
            str(sdk2.ResourceData(self.Parameters.personal_info_patcher).path),
            '--ydb_host', self.Parameters.ydb_host,
            '--ydb_port', self.Parameters.ydb_port,
            '--ydb_database', self.Parameters.ydb_database,
            '--ydb_home', self.Parameters.ydb_home,
            '--personal_info_field', '_messenger_guid'
        ]
        self.execute_cmd(patch_personal_infos, 'patch_personal_info', inp_file='puid2guid', err_file='patch.errors', env=env)

    def on_execute(self):
        self.register_workers()
