import os
import logging

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from sandbox import sdk2

from sandbox.projects.ydo import get_now_utc, get_id_or_none, YdoFeaturesJson
from sandbox.projects.ydo.moderation import YdoModerationOpsExecutable
from sandbox.projects.ydo.moderation.GetUsersBatch import YdoModerationGetUsersBatch
from sandbox.projects.ydo.moderation.ExportFromKikimr import YdoModerationExportFromKikimr
from sandbox.projects.ydo.moderation.PrepareRows import YdoModerationPrepareRows
from sandbox.projects.ydo.moderation.UploadExport import YdoModerationUploadExport
from sandbox.projects.ydo.yt_table_to_lb import YdoYtTableToLb, YdoYtToLbExecutable


class YdoModerationExportConveyor(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")

        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="ydb-token", required=True)
            ydb_host = sdk2.parameters.String("Host", required=True)
            ydb_port = sdk2.parameters.String("Port", required=True)
            ydb_database = sdk2.parameters.String("Database", required=True)
            ydb_home = sdk2.parameters.String("Home", required=True)

        with sdk2.parameters.Group("Ydo parameters") as ydo_block:
            use_stable_resources = sdk2.parameters.Bool(
                "Use stable resources?",
                default=True,
            )
            with use_stable_resources.value[False]:
                ydo_moderation_ops = sdk2.parameters.Resource(
                    "ydo_moderation_ops",
                    resource_type=YdoModerationOpsExecutable,
                    required=True,
                )
                features = sdk2.parameters.Resource(
                    "features",
                    resource_type=YdoFeaturesJson,
                    required=True,
                )

        with sdk2.parameters.Group("GetUsersBatch parameters") as get_user_batch_block:
            yt_backup_folder = sdk2.parameters.String("Folder", required=True)
            batch_size = sdk2.parameters.Integer("Batch size", default=1000, required=True)
            worker_id_pattern = sdk2.parameters.String("worker_id regexp pattern for sending to moderation")

        with sdk2.parameters.Group("ExportFromKikimr parameters") as export_from_kikimr_block:
            wait_interval = sdk2.parameters.Integer(
                "Moderation wait interval",
                required=True
            )

        with sdk2.parameters.Group("UploadExport parameters") as upload_export_block:
            export_folder = sdk2.parameters.String("Export folder", required=True)
            org_export_folder = sdk2.parameters.String("Org export folder", required=True)
            approve_folder = sdk2.parameters.String("Approve folder", required=True)

        with sdk2.parameters.Group("LogbrokerExport parameters") as logbroker_export_block:
            lb_use_stable_resources = sdk2.parameters.Bool(
                "Use stable resources?",
                default=True,
            )
            with lb_use_stable_resources.value[False]:
                yt_to_lb = sdk2.parameters.Resource(
                    "ydo_yt_to_lb",
                    resource_type=YdoYtToLbExecutable,
                    required=False,
                )

            send_export_to_lb = sdk2.parameters.Bool("Send export to logbroker", default=False)
            tvm_vault_secret = sdk2.parameters.String("Your tvm secret name in vault", required=False)
            ydo_tvm_id = sdk2.parameters.Integer("YDO TVM id", required=False)
            logbroker_topic = sdk2.parameters.String("Logbroker topic", required=False)
            logbroker_batch_size = sdk2.parameters.Integer("Logbroker batch size", required=False, default=10)
            fail_on_exception = sdk2.parameters.Bool("Fail on exception", default=False)
            raise_on_writes_errors = sdk2.parameters.Bool("Raise on writes errors", default=False)
            add_message_timestamp = sdk2.parameters.Bool("Add send timestamp to message", default=False)
            timestamp_field = sdk2.parameters.String("Field name for timestamp in message", default='export_ts')
            lb_cooldown = sdk2.parameters.Float("Time for Logbroker to cooldown")
            lb_use_gzip_compression = sdk2.parameters.Bool("Use GZIP compression", default=False)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def check_exit(self):
        return self.Context.exit

    def exit(self, wait_time):
        self.Context.exit = True
        raise sdk2.WaitTime(wait_time)

    def init_context(self):
        self.Context.exit = False
        self.Context.timestamp = get_now_utc()
        self.Context.export_result_tables = []

    def get_kikimr_dict(self):
        return dict(
            kikimr_host=self.Parameters.ydb_host,
            kikimr_port=self.Parameters.ydb_port,
            kikimr_database=self.Parameters.ydb_database,
            kikimr_home=self.Parameters.ydb_home,
        )

    def get_kikimr_secret_env(self):
        result = dict()
        if self.Parameters.ydb_host != 'ydb-services.yandex.net':
            result['YDB_TOKEN'] = self.Parameters.ydb_vault_token
        return result

    def get_user_batch(self):
        kikimr_dict = self.get_kikimr_dict()
        kikimr_home = kikimr_dict.pop('kikimr_home')
        kikimr_dict['kikimr_table'] = os.path.join(kikimr_home, 'workers_to_moderation')

        task = YdoModerationGetUsersBatch(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            release_chooser='stable' if self.Parameters.use_stable_resources else 'custom',
            moderation_ops=get_id_or_none(self.Parameters.ydo_moderation_ops),
            batch_size=self.Parameters.batch_size,
            worker_id_pattern=self.Parameters.worker_id_pattern,
            env=dict(YT_PROXY=self.Parameters.yt_host),
            secret_env=dict(YT_TOKEN=self.Parameters.yt_vault_token, **self.get_kikimr_secret_env()),
            yt_backup_folder=self.Parameters.yt_backup_folder,
            **kikimr_dict
        )

        task.enqueue()

        self.Context.get_user_batch_task_id = task.id

        raise sdk2.WaitTask([self.Context.get_user_batch_task_id], ctt.Status.Group.SUCCEED, wait_all=True)

    def check_get_user_batch(self):
        if not sdk2.Task[self.Context.get_user_batch_task_id].Parameters.result_id:
            return "Empty user batch", True
        return "", False

    def export_from_kikimr(self):
        kikimr_dict = self.get_kikimr_dict()

        task = YdoModerationExportFromKikimr(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            release_chooser='stable' if self.Parameters.use_stable_resources else 'custom',
            moderation_ops=get_id_or_none(self.Parameters.ydo_moderation_ops),
            features=get_id_or_none(self.Parameters.features),
            env=dict(),
            secret_env=dict(**self.get_kikimr_secret_env()),
            users_batch=sdk2.Task[self.Context.get_user_batch_task_id].Parameters.result_id,
            timestamp=self.Context.timestamp,
            wait_interval=self.Parameters.wait_interval,
            **kikimr_dict
        )

        task.enqueue()

        self.Context.export_from_kikimr_task_id = task.id

        raise sdk2.WaitOutput({self.Context.export_from_kikimr_task_id: ("data_file_id", "commit_file_id")}, wait_all=True)

    def retryable_prepare_rows(self):
        if self.Context.prepare_rows_task_id is ctm.NotExists or sdk2.Task[self.Context.prepare_rows_task_id].status != ctt.Status.SUCCESS:
            return self.prepare_rows()

    def prepare_rows(self):
        task = YdoModerationPrepareRows(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            release_chooser='stable' if self.Parameters.use_stable_resources else 'custom',
            moderation_ops=get_id_or_none(self.Parameters.ydo_moderation_ops),
            features=get_id_or_none(self.Parameters.features),
            data_file=sdk2.Task[self.Context.export_from_kikimr_task_id].Parameters.data_file_id,
            commit_file=sdk2.Task[self.Context.export_from_kikimr_task_id].Parameters.commit_file_id,
        )

        task.enqueue()

        self.Context.prepare_rows_task_id = task.id

        raise sdk2.WaitOutput({self.Context.prepare_rows_task_id: ("export_file_id",)}, wait_all=True)

    def make_upload_export_task(self, resource, table):
        return YdoModerationUploadExport(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            data_file=resource.id,
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_host=self.Parameters.yt_host,
            yt_table=table,
        )

    def upload_export(self):
        tasks = list()

        prepare_rows_task = sdk2.Task[self.Context.prepare_rows_task_id]
        export_resource = sdk2.Resource[prepare_rows_task.Parameters.export_file_id]
        org_export_resource = sdk2.Resource[prepare_rows_task.Parameters.org_export_file_id]
        approve_resource = sdk2.Resource[prepare_rows_task.Parameters.approve_file_id]

        if export_resource.line_count != 0:
            export_table_path = os.path.join(self.Parameters.export_folder, str(self.Context.timestamp))
            self.Context.export_result_tables.append(export_table_path)
            tasks.append(self.make_upload_export_task(export_resource, export_table_path))

        if org_export_resource.line_count != 0:
            org_export_table_path = os.path.join(self.Parameters.org_export_folder, str(self.Context.timestamp))
            self.Context.export_result_tables.append(org_export_table_path)
            tasks.append(self.make_upload_export_task(org_export_resource, org_export_table_path))

        if approve_resource.line_count != 0:
            tasks.append(self.make_upload_export_task(approve_resource, os.path.join(self.Parameters.approve_folder, str(self.Context.timestamp))))

        if tasks:
            task_ids = list()
            for task in tasks:
                task_ids.append(task.id)
                task.enqueue()
            raise sdk2.WaitTask(task_ids, ctt.Status.Group.SUCCEED, wait_all=True)

    def make_send_to_lb_task(self, yt_path):
        return YdoYtTableToLb(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            ydo_yt_to_lb=get_id_or_none(self.Parameters.ydo_moderation_ops),
            yt_path=yt_path,
            logbroker_topic=self.Parameters.logbroker_topic,
            logbroker_batch_size=self.Parameters.logbroker_batch_size,
            client_tvm_id=self.Parameters.ydo_tvm_id,
            fail_on_exception=self.Parameters.fail_on_exception,
            raise_on_writes_errors=self.Parameters.raise_on_writes_errors,
            add_message_timestamp=self.Parameters.add_message_timestamp,
            timestamp_field=self.Parameters.timestamp_field,
            lb_cooldown=self.Parameters.lb_cooldown,
            lb_use_gzip_compression=self.Parameters.lb_use_gzip_compression,
            env=dict(YT_PROXY=self.Parameters.yt_host),
            secret_env=dict(
                YT_TOKEN=self.Parameters.yt_vault_token,
                LOGBROKER_TVM_SECRET=self.Parameters.tvm_vault_secret,
            ),
        )

    def send_export_to_lb(self):
        for yt_path in self.Context.export_result_tables:
            self.make_send_to_lb_task(yt_path).enqueue()

    def on_execute(self):
        with self.memoize_stage.init_context:
            self.init_context()

        if self.check_exit():
            return

        with self.memoize_stage.get_user_batch:
            self.get_user_batch()

        with self.memoize_stage.check_get_user_batch:
            message, status = self.check_get_user_batch()
            if status:
                logging.info(message)
                self.exit(60 * 60)

        with self.memoize_stage.export_from_kikimr:
            self.export_from_kikimr()

        with self.memoize_stage.prepare_rows(max_runs=5, commit_on_entrance=False):
            self.retryable_prepare_rows()

        with self.memoize_stage.upload_export():
            self.upload_export()
            logging.info("Nothing to upload")
            self.exit(60 * 60)

        if self.check_exit():
            logging.info("Nothing to send to LB")
            return

        with self.memoize_stage.send_export_to_lb():
            if self.Parameters.send_export_to_lb:
                self.send_export_to_lb()
