import os
import json
import logging

from sandbox.common.errors import TaskError
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog

from sandbox.projects.common import solomon

from sandbox.projects.ydo import get_last_released_resource
from sandbox.projects.ydo.moderation import YdoModerationBaseTask, YdoModerationDataFile, YdoModerationOpsExecutable


class YdoModerationGetYangImport(YdoModerationBaseTask):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("YT parameters") as yt_block:
            push_monitoring = sdk2.parameters.Bool(
                "Push monitoring?",
                default=True,
            )
            use_stable_resources = sdk2.parameters.Bool(
                "Use stable resources?",
                default=True,
            )
            with use_stable_resources.value[False]:
                ydo_moderation_ops_executable_resource = sdk2.parameters.Resource(
                    "ydo_moderation_ops",
                    resource_type=YdoModerationOpsExecutable,
                    required=True,
                )
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")
            yt_lock_node = sdk2.parameters.String("Lock path", required=True)
            yt_import_folder = sdk2.parameters.String("Folder", required=True)
            yt_bad_import_folder = sdk2.parameters.String("Bad import folder", required=True)
            yt_backup_folder = sdk2.parameters.String("Backup folder", required=True)
            yt_export_folder = sdk2.parameters.String("Export folder", required=True)
            yt_org_export_folder = sdk2.parameters.String("Org export folder", required=True)
            yt_export_backup_folder = sdk2.parameters.String("Export backup folder", required=True)

        with sdk2.parameters.Output:
            result_id = sdk2.parameters.Integer("Yang import resource_id")

    def get_yt_parameters(self):
        return sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token), self.Parameters.yt_host

    def validate_and_move_bad(self, moderation_ops, yt_client, table):
        if table.endswith("_automatic") or table.endswith("_hand"):
            return True
        export_table, _ = self.get_paired_export_table(table)
        import_table = os.path.join(self.Parameters.yt_import_folder, table)
        bad_table = os.path.join(self.Parameters.yt_bad_import_folder, table)
        if export_table is None or yt_client.row_count(import_table) == 0:
            yt_client.move(import_table, bad_table)
            return False

        result_file = "validation_result.json"
        cmd = [
            moderation_ops, "validate_yang",
            "--export-table", export_table,
            "--import-table", import_table,
            "--output", result_file
        ]

        with ProcessLog(self, logger="moderation_ops") as pl:
            env = os.environ.copy()
            env["YT_TOKEN"], env["YT_PROXY"] = self.get_yt_parameters()
            run = sp.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
            run.communicate()

            assert run.returncode == 0

        with open(result_file) as validation_result:
            result = json.load(validation_result)

        logging.info(import_table)
        logging.info(result)

        if result["status"] == "OK":
            return True
        else:
            yt_client.move(import_table, bad_table)
            return False

    def get_next_tables(self, yt_client):
        moderation_ops = str(get_last_released_resource(
            YdoModerationOpsExecutable,
            condition=self.Parameters.use_stable_resources,
            default=self.Parameters.ydo_moderation_ops_executable_resource,
            error_msg="YdoModerationOpsExecutable not founded",
        ).path)

        tables = list(yt_client.list(self.Parameters.yt_import_folder))
        bad_table_count = 0
        result_tables = list()
        for table in tables:
            if self.validate_and_move_bad(moderation_ops, yt_client, table):
                result_tables.append(table)
            else:
                bad_table_count += 1

        if self.Parameters.push_monitoring:
            try:
                token = sdk2.Vault.data(self.owner, 'solomon-token')
                solomon.push_to_solomon_v2(
                    token,
                    dict(
                        project="ydo",
                        cluster="moderation",
                        service="simple_monitoring",
                    ),
                    solomon.create_sensors(dict(bad_table_count=bad_table_count))
                )
            except Exception:
                if bad_table_count:
                    raise

        return result_tables

    def get_paired_export_table(self, table):
        timestamp, type = table.split("_", 1)
        if type == "org_yang":
            export_table = os.path.join(self.Parameters.yt_org_export_folder, timestamp)
            result_table = os.path.join(self.Parameters.yt_export_backup_folder, table)
        elif type == "yang":
            export_table = os.path.join(self.Parameters.yt_export_folder, timestamp)
            result_table = os.path.join(self.Parameters.yt_export_backup_folder, table)
        else:
            export_table = None
            result_table = None

        return export_table, result_table

    def read_from_yt(self, yt_client):
        next_tables = self.get_next_tables(yt_client)
        if not next_tables:
            return

        result = YdoModerationDataFile(self, "from task: {}".format(self.id), "import.ysonl")
        result_data = sdk2.ResourceData(result)
        env = os.environ.copy()
        env["YT_TOKEN"], env["YT_PROXY"] = self.get_yt_parameters()

        rows_count = 0
        for i, next_table in enumerate(next_tables):
            move_table = os.path.join(self.Parameters.yt_import_folder, next_table)

            with ProcessLog(self, logger="read_from_yt") as read_from_yt_pl, ProcessLog(self, logger="remove_semicolon") as remove_semicolon_pl:
                read_from_yt_cmd = [
                    "yt", "read",
                    move_table,
                    "--format", "<format=text>yson"
                ]
                remove_semicolon_cmd = ["sed", "s/;$//"]

                with open(str(result_data.path), "a") as result_file:
                    read_from_yt = sp.Popen(read_from_yt_cmd, stdout=sp.PIPE, stderr=read_from_yt_pl.stderr, env=env)
                    remove_semicolon = sp.Popen(remove_semicolon_cmd, stdin=read_from_yt.stdout, stdout=result_file, stderr=remove_semicolon_pl.stderr)

                    remove_semicolon.communicate()

                if remove_semicolon.returncode != 0:
                    raise TaskError("Process read_from_yt has failed")

                if i + 1 < len(next_tables):
                    logging.info("Add endl to result file")
                    with open(str(result_data.path), "a") as result_file:
                        result_file.write('\n')

                rows_count += yt_client.row_count(move_table)

        assert rows_count == self.wc_lines(str(result_data.path))

        for next_table in next_tables:
            move_table = os.path.join(self.Parameters.yt_import_folder, next_table)
            backup_table = os.path.join(self.Parameters.yt_backup_folder, next_table)

            yt_client.move(move_table, backup_table)
            export_move_table, export_backup_table = self.get_paired_export_table(next_table)
            if export_move_table is not None:
                yt_client.move(export_move_table, export_backup_table)

        result.line_count = rows_count
        result_data.ready()
        return result.id

    def on_execute(self):
        import yt.wrapper as yt
        yt.config["token"], yt.config["proxy"]["url"] = self.get_yt_parameters()

        with yt.Transaction():
            yt.lock(self.Parameters.yt_lock_node, waitable=True)

            resource_id = self.read_from_yt(yt)
            if resource_id is not None:
                self.Parameters.result_id = resource_id
                return
            logging.info("Result is empty")
