import os
import logging

import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects.geosearch.CleanupYtFolder import CleanupYtFolder

from sandbox.projects.ydo import get_now_utc, get_id_or_none, YdoFeaturesJson
from sandbox.projects.ydo.moderation import (
    YdoModerationOpsExecutable,
    YdoModerationSearchAsyncApiConfig,
)

from sandbox.projects.ydo.backup.MergeTables import YdoBackupMergeTables

from sandbox.projects.ydo.moderation.GetYangImport import YdoModerationGetYangImport
from sandbox.projects.ydo.moderation.ImportFromYang import YdoModerationImportFromYang
from sandbox.projects.ydo.moderation.UploadExport import YdoModerationUploadExport


class YdoModerationImportConveyor(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_vault_token = sdk2.parameters.String("Your yt token name in vault", default="yt-token", required=True)
            with sdk2.parameters.RadioGroup("Host") as yt_host:
                yt_host.values["hahn"] = yt_host.Value(value="Hahn", default=True)
                yt_host.values["banach"] = yt_host.Value(value="Banach")

        with sdk2.parameters.Group("YDB parameters") as ydb_block:
            ydb_vault_token = sdk2.parameters.String("Your ydb token name in vault", default="ydb-token", required=True)
            ydb_host = sdk2.parameters.String("Host", required=True)
            ydb_port = sdk2.parameters.String("Port", required=True)
            ydb_database = sdk2.parameters.String("Database", required=True)
            ydb_home = sdk2.parameters.String("Home", required=True)

        with sdk2.parameters.Group("Ydo parameters") as ydo_block:
            use_stable_resources = sdk2.parameters.Bool(
                "Use stable resources?",
                default=True,
            )
            with use_stable_resources.value[False]:
                ydo_moderation_ops = sdk2.parameters.Resource(
                    "ydo_moderation_ops",
                    resource_type=YdoModerationOpsExecutable,
                    required=True,
                )
                features = sdk2.parameters.Resource(
                    "features",
                    resource_type=YdoFeaturesJson,
                    required=True,
                )
                search_async_api_config = sdk2.parameters.Resource(
                    'Search Async Api config',
                    resource_type=YdoModerationSearchAsyncApiConfig,
                    required=True,
                )

        with sdk2.parameters.Group("GetYangImport parameters") as get_yang_import_block:
            yt_lock_node = sdk2.parameters.String("Lock path", required=True)
            yt_import_folder = sdk2.parameters.String("Folder", required=True)
            yt_bad_import_folder = sdk2.parameters.String("Bad import folder", required=True)
            yt_backup_folder = sdk2.parameters.String("Backup folder", required=True)
            yt_export_folder = sdk2.parameters.String("Export folder", required=True)
            yt_org_export_folder = sdk2.parameters.String("Org export folder", required=True)
            yt_export_backup_folder = sdk2.parameters.String("Export backup folder", required=True)

        with sdk2.parameters.Group("ImportFromYang parameters") as import_from_yang_block:
            use_rt_indexer = sdk2.parameters.Bool('Use RT indexer', default=False)
            logbroker_vault_token = sdk2.parameters.String("Your logbroker token name in vault", default="LOGBROKER_TOKEN", required=False)

        with sdk2.parameters.Group("UploadResults parameters") as upload_export_block:
            filter_folder = sdk2.parameters.String("Filter folder", required=True)
            org_filter_folder = sdk2.parameters.String("Org filter folder", required=True)

        with sdk2.parameters.Group("Merging parameters") as merging_block:
            history_size = sdk2.parameters.Integer("History size", default=20, required=True)
            merge_folder = sdk2.parameters.String("Folder for merged tables", required=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def init_context(self):
        self.Context.exit = False
        self.Context.folder_paths = (self.Parameters.yt_backup_folder, self.Parameters.yt_export_backup_folder,
                                     self.Parameters.filter_folder)

    def check_exit(self):
        return self.Context.exit

    def exit(self, wait_time):
        self.Context.exit = True
        raise sdk2.WaitTime(wait_time)

    def get_kikimr_dict(self):
        return dict(
            kikimr_host=self.Parameters.ydb_host,
            kikimr_port=self.Parameters.ydb_port,
            kikimr_database=self.Parameters.ydb_database,
            kikimr_home=self.Parameters.ydb_home,
        )

    def get_kikimr_secret_env(self):
        result = dict()
        if self.Parameters.ydb_host != 'ydb-services.yandex.net':
            result['YDB_TOKEN'] = self.Parameters.ydb_vault_token
        return result

    def check_yang_import(self):
        if not sdk2.Task[self.Context.get_yang_import_task_id].Parameters.result_id:
            return "Empty yang import", True
        return "", False

    def get_yang_import(self):
        task = YdoModerationGetYangImport(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            use_stable_resources=self.Parameters.use_stable_resources,
            ydo_moderation_ops_executable_resource=get_id_or_none(self.Parameters.ydo_moderation_ops),
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_host=self.Parameters.yt_host,
            yt_lock_node=self.Parameters.yt_lock_node,
            yt_import_folder=self.Parameters.yt_import_folder,
            yt_bad_import_folder=self.Parameters.yt_bad_import_folder,
            yt_backup_folder=self.Parameters.yt_backup_folder,
            yt_export_folder=self.Parameters.yt_export_folder,
            yt_org_export_folder=self.Parameters.yt_org_export_folder,
            yt_export_backup_folder=self.Parameters.yt_export_backup_folder,
        )

        task.enqueue()

        self.Context.get_yang_import_task_id = task.id

        raise sdk2.WaitTask([self.Context.get_yang_import_task_id], ctt.Status.Group.SUCCEED, wait_all=True)

    def import_from_yang(self):
        secret_env = dict(**self.get_kikimr_secret_env())
        secret_env['LOGBROKER_TOKEN'] = self.Parameters.logbroker_vault_token

        task = YdoModerationImportFromYang(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            release_chooser='stable' if self.Parameters.use_stable_resources else 'custom',
            moderation_ops=get_id_or_none(self.Parameters.ydo_moderation_ops),
            features=get_id_or_none(self.Parameters.features),
            env=dict(),
            secret_env=secret_env,
            input_file=sdk2.Task[self.Context.get_yang_import_task_id].Parameters.result_id,
            use_rt_indexer=self.Parameters.use_rt_indexer,
            search_async_api_config=self.Parameters.search_async_api_config,
            **self.get_kikimr_dict()
        )

        task.enqueue()

        self.Context.import_from_yang_task_id = task.id

        raise sdk2.WaitOutput({self.Context.import_from_yang_task_id: ("error_file_id",)}, wait_all=True)

    def make_upload_export_task(self, resource, table):
        return YdoModerationUploadExport(
            self,
            description="Run for {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            data_file=resource.id,
            yt_vault_token=self.Parameters.yt_vault_token,
            yt_host=self.Parameters.yt_host,
            yt_table=table,
        )

    def upload_mail(self):
        tasks = list()

        import_from_yang_task = sdk2.Task[self.Context.import_from_yang_task_id]
        filter_resource = sdk2.Resource[import_from_yang_task.Parameters.filter_file_id]
        org_filter_resource = sdk2.Resource[import_from_yang_task.Parameters.org_filter_file_id]

        if filter_resource.line_count != 0:
            tasks.append(self.make_upload_export_task(filter_resource, os.path.join(self.Parameters.filter_folder, str(get_now_utc()))))

        if org_filter_resource.line_count != 0:
            tasks.append(self.make_upload_export_task(org_filter_resource, os.path.join(self.Parameters.org_filter_folder, str(get_now_utc()))))

        if tasks:
            task_ids = list()
            for task in tasks:
                task_ids.append(task.id)
                task.enqueue()
            raise sdk2.WaitTask(task_ids, ctt.Status.Group.SUCCEED, wait_all=True)

    def check_errors(self):
        import_from_yang_task = sdk2.Task[self.Context.import_from_yang_task_id]
        error_resource = sdk2.Resource[import_from_yang_task.Parameters.error_file_id]

        assert error_resource.line_count == 0

    def merge_folders(self):
        tasks = [
            YdoBackupMergeTables(
                self,
                description="Merge old tables for task {}".format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_host=self.Parameters.yt_host,
                yt_vault_token=self.Parameters.yt_vault_token,
                path=folder_path,
                history_size=self.Parameters.history_size,
                merged_path=os.path.join(self.Parameters.merge_folder, os.path.basename(folder_path))
            )
            for folder_path in self.Context.folder_paths
        ]
        for task in tasks:
            task.enqueue()

        raise sdk2.WaitTask([task.id for task in tasks], ctt.Status.Group.SUCCEED, wait_all=True)

    def clean_folders(self):
        tasks = [
            CleanupYtFolder(
                self,
                description='Remove old tables for task {}'.format(self.id),
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                yt_host=self.Parameters.yt_host,
                yt_vault_token=self.Parameters.yt_vault_token,
                path=folder_path,
                history_size=self.Parameters.history_size,
            )
            for folder_path in self.Context.folder_paths
        ]
        for task in tasks:
            task.enqueue()

        raise sdk2.WaitTask([task.id for task in tasks], ctt.Status.Group.SUCCEED, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.init_context:
            self.init_context()

        if self.check_exit():
            return

        with self.memoize_stage.get_yang_import:
            self.get_yang_import()

        with self.memoize_stage.check_yang_import:
            message, status = self.check_yang_import()
            if status:
                logging.info(message)
                self.exit(10 * 60)

        with self.memoize_stage.import_from_yang:
            self.import_from_yang()

        with self.memoize_stage.upload_mail:
            self.upload_mail()

        with self.memoize_stage.check_errors:
            self.check_errors()

        with self.memoize_stage.merge_folders:
            self.merge_folders()

        with self.memoize_stage.clean_folders:
            self.clean_folders()
