import logging
import tempfile
import time

from sandbox import sdk2

from sandbox.projects.common import file_utils

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.default_multislot_task import YdoMutlihostTask
from sandbox.projects.ydo.moderation import YdoModerationOpsExecutable
from sandbox.projects.ydo.parameters_view import generate_view, with_config


class YdoWorkersModerationScheduler(YdoMutlihostTask):
    class Parameters(
        with_config(
            generate_view(
                moderation_ops=YdoModerationOpsExecutable,
                check_owner=True,
                with_environ=True,
                default_secret_env_vars={
                    "YDB_TOKEN": "ydb-token",
                    "LOGBROKER_TOKEN": "LOGBROKER_TOKEN",
                },
            ),
            "ydo/async_api/client_configs",
            [
                ("production", "production/async_api_pipe_template.yaml"),
                ("testing", "testing/async_api_pipe_template.yaml"),
            ],
        )
    ):
        with sdk2.parameters.Group("Kikimr parameters") as kikimr_block:
            kikimr_host = sdk2.parameters.String("Host", required=True)
            kikimr_port = sdk2.parameters.String("Grpc port", required=True)
            kikimr_database = sdk2.parameters.String("Database", required=True)
            kikimr_home = sdk2.parameters.String("Home", required=True)

        wait_interval = sdk2.parameters.Integer("Moderation wait interval", required=True)
        logbroker_source = sdk2.parameters.String(
            "Source for Logbroker", required=True, default="YdoWorkersModerationScheduler"
        )
        worker_id_pattern = sdk2.parameters.String("worker_id regexp pattern for sending to moderation")

    def on_execute(self):
        self.Context.now_ts = int(time.time())

        with open(self.Parameters.config_path) as f_config:
            raw_config = f_config.read()

        logging.info('raw_config: {}'.format(raw_config))
        filled_config = raw_config.replace('{{ SOURCE }}', self.Parameters.logbroker_source)
        logging.info('filled_config: {}'.format(filled_config))
        filled_config_tempfile = tempfile.NamedTemporaryFile()
        file_utils.write_file(filled_config_tempfile.name, filled_config)
        filled_config_tempfile.seek(0)

        cmd = [
            self.Parameters.moderation_ops_path,
            "async_api_export",
            "--ydb_host",
            self.Parameters.kikimr_host,
            "--ydb_port",
            self.Parameters.kikimr_port,
            "--ydb_database",
            self.Parameters.kikimr_database,
            "--ydb_home",
            self.Parameters.kikimr_home,
            "--timestamp",
            str(self.Context.now_ts),
            "--wait-interval",
            str(self.Parameters.wait_interval),
            "--async_api_config",
            filled_config_tempfile.name,
        ] + (["--worker_id_pattern", self.Parameters.worker_id_pattern] if self.Parameters.worker_id_pattern else [])

        execute_cmd(cmd, "ydo_moderation_ops async_api_export", "Export failed", env=self.Parameters.get_environ())
