# -*- coding: utf-8 -*-
from datetime import date, timedelta, datetime
import logging

import sandbox.sdk2 as sdk2

from sandbox.common.errors import TaskError
from sandbox.projects.common import solomon
from sandbox.projects.common.yql import run_batch
from sandbox.sandboxsdk.environments import PipEnvironment

QUERY_TEMP = '''
INSERT INTO @wins
SELECT
    query,
    region,
    SUM(vsurplus["{surplus_version}"]["wiz-uslugi"].click_win) AS wins
FROM
    `//home/ydo/service/storage/surplus/{table}`
GROUP BY
    query, region
ORDER BY
    wins DESC
LIMIT 2000;

INSERT INTO
    @ans
SELECT
    query,
    region,
    COUNT_IF(DictContains(search_props, "UPPER.YdoWizard.NumberOfOffers")) AS answer_count,
    COUNT_IF(NOT DictContains(search_props, "UPPER.YdoWizard.NumberOfOffers")) AS no_answer_count
FROM
    `//home/geo-search/alextim27/uslugi/production/squeeze/search/daily/{table}`
GROUP BY
    query, region;

COMMIT;

INSERT INTO `{result_dir}/{table}` WITH TRUNCATE
SELECT
    wins.*,
    ans.*,
    100 * CAST(no_answer_count as DOUBLE) / (no_answer_count + answer_count) as no_answer_perc
FROM
    @ans as ans INNER JOIN
    @wins as wins
USING
    (query, region)
ORDER BY wins DESC, answer_count DESC
LIMIT 1000;
'''


class YdoNoAnswersStat(sdk2.Task):
    """
        Подсчёт ежедневной статистики неответов в Услугах по частотникам
    """

    class Parameters(sdk2.Parameters):
        result_dir = sdk2.parameters.String(
            'Directory with no_answer daily statistics',
            default_value='//home/ydo/stat/no_answer_daily'
        )

        date_str_from = sdk2.parameters.String(
            'Date from',
            default_value=(date.today()-timedelta(1)).isoformat()
        )
        date_str_to = sdk2.parameters.String(
            'Date to',
            default_value=(date.today()-timedelta(1)).isoformat()
        )

        surplus_version = sdk2.parameters.String(
            'Surplus version',
            default_value='v7'
        )

        vault_token_owner = sdk2.parameters.String(
            'Vault owner for YQL and YT tokens', required=True
        )
        yt_vault_token_name = sdk2.parameters.String(
            'Vault name for YT token', required=True
        )
        yql_vault_token_name = sdk2.parameters.String(
            'Vault name for YQL token', required=True
        )

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = [
            PipEnvironment('yandex-yt'),
            PipEnvironment('yql')
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        import yt.wrapper as yt
        yt.config['proxy']['url'] = 'hahn'
        yt.config['token'] = sdk2.Vault.data(self.Parameters.vault_token_owner, name=self.Parameters.yt_vault_token_name)

        date_from = datetime.strptime(self.Parameters.date_str_from, '%Y-%m-%d').date()
        date_to = datetime.strptime(self.Parameters.date_str_to, '%Y-%m-%d').date()
        date_range = (date_to - date_from).days + 1

        batch = []
        for day in range(date_range):
            date_to_process = (date_from + timedelta(day)).isoformat()
            date_without = date_to_process.replace('-', '')
            yt_path = '/'.join((self.Parameters.result_dir, date_to_process))
            if not yt.exists(yt_path):
                query = QUERY_TEMP.format(
                    result_dir=self.Parameters.result_dir,
                    table=date_to_process,
                    table_without=date_without,
                    surplus_version=self.Parameters.surplus_version
                )
                batch.append((yt_path, query))

        error = ''
        try:
            run_batch(batch, sdk2.Vault.data(self.Parameters.vault_token_owner, name=self.Parameters.yql_vault_token_name), 'hahn')
        except TaskError as err:
            error = err

        stats = dict()
        for day in range(date_range):
            date_to_process = (date_from + timedelta(day)).isoformat()
            yt_path = '/'.join((self.Parameters.result_dir, date_to_process))
            if yt.exists(yt_path):
                no_answer_perc_list = [row['no_answer_perc'] for row in yt.read_table(yt.TablePath(yt_path, columns=['no_answer_perc']), format='json')]
                stats_daily = dict()
                for top_size in (5, 25, 100, 500):
                    stat_name = 'ydo_no_ans_top{}_perc'.format(top_size)
                    stats_daily[stat_name] = sum(no_answer_perc_list[:top_size]) / top_size
                stats[date_to_process] = stats_daily
        self.push_results(stats)
        if error:
            raise error

    def push_results(self, stats):
        """Send sensor values to solomon"""
        logging.info('stats is %s', stats)
        common_labels = {
            'project': 'ydo_stat',
            'cluster': 'no_answer',
            'service': 'no_answer'
        }
        sensors = []
        for date_str, stat in stats.items():
            sensors_daily = solomon.create_sensors(stat)
            time_label = datetime.strptime(date_str, '%Y-%m-%d') + timedelta(hours=12)
            ts = int(time_label.strftime('%s'))
            for sensor in sensors_daily:
                sensor['ts'] = ts
                sensors.append(sensor)
        logging.info('upload to solomon \ncommon_labels:{labels} \nsensors : {sensors}'.format(labels=common_labels, sensors=sensors))
        token = sdk2.Vault.data(self.Parameters.vault_token_owner, 'solomon-token')
        solomon.push_to_solomon_v2(token, common_labels, sensors)
