from sandbox import sdk2, common
from sandbox.projects.common.nanny import nanny
import sandbox.common.types.task as ctt

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoFeaturesJson,
    YdoOrderCustomizerJson,
    YdoTestingOrderCustomizerJson,
    YdoRawOrderCustomizerJson,
)
from sandbox.projects.ydo.order_clarifier.ReleaseYdoFormAnswersGzt import YdoReleaseFormAnswersGzt
from sandbox.projects.ydo.order_clarifier.ReleaseOrderCustomizerAdditionals import YdoReleaseOrderCustomizerAdditionals
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.ydo.self_release.ReleaseParentTask import (
    SelfReleaseOnSuccessMixin,
    SelfReleaseOnSuccessMixinParameters,
    YdoReleaseTask,
)


class YdoOrderClarifierConfigBuilderExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoReleaseOrderCustomizerJson(SelfReleaseOnSuccessMixin, nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(
        generate_view(
            ydo_order_clarifier_config_builder=YdoOrderClarifierConfigBuilderExecutable,
            check_owner=True,
        )
    ):
        self_release_parameters = SelfReleaseOnSuccessMixinParameters()

        input_config_raw = sdk2.parameters.String('Input config', required=True)
        input_config_as_diff = sdk2.parameters.Bool('Treat "Input config" as diff')
        base_config_raw_resource = sdk2.parameters.Resource(
            'Base config resource',
            resource_type=YdoRawOrderCustomizerJson,
            required=True,
        )
        additional_cmd_params = sdk2.parameters.String('Additional cmd args')
        build_and_release_form_answers_gzt = sdk2.parameters.Bool(
            'Build and release form answers gazetteer in the end', default=True
        )
        build_and_release_order_customizer_additionals = sdk2.parameters.Bool(
            'Build and release order customizer additionals in the end', default=True
        )

        with sdk2.parameters.Output:
            merge_conflict_dump = sdk2.parameters.String('Raw config merge errors')
            validation_error_dump_testing = sdk2.parameters.String('Testing validation errors')
            validation_error_dump_production = sdk2.parameters.String('Production validation errors')

    class Context(sdk2.Task.Context):
        order_customizer_json_prod_id = None
        order_customizer_json_testing_id = None
        rubrics_json_prod_id = None
        rubrics_json_testing_id = None
        features_json_prod_id = None
        features_json_testing_id = None
        form_answers_gzt_prod_task_id = None
        form_answers_gzt_testing_task_id = None

    class Requirements(sdk2.Requirements):
        semaphores = ctt.Semaphores(
            acquires=[ctt.Semaphores.Acquire(name='release_order_customizer_json_queue')],
            release=(ctt.Status.RELEASED, ctt.Status.Group.BREAK, ctt.Status.FAILURE),
        )
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        order_customizer_output_raw_resource = YdoRawOrderCustomizerJson(
            self, 'order_customizer_raw.json', 'order_customizer_raw.json'
        )
        order_customizer_output_raw_resource_data = sdk2.ResourceData(order_customizer_output_raw_resource)
        order_customizer_testing_resource = YdoTestingOrderCustomizerJson(
            self, 'order_customizer_testing.json', 'order_customizer_testing.json'
        )
        order_customizer_testing_resource_data = sdk2.ResourceData(order_customizer_testing_resource)
        order_customizer_production_resource = YdoOrderCustomizerJson(
            self, 'order_customizer_production.json', 'order_customizer_production.json'
        )
        order_customizer_production_resource_data = sdk2.ResourceData(order_customizer_production_resource)

        with open('input_order_customizer_raw.json', 'w') as f:
            f.write(self.Parameters.input_config_raw)

        rubrics_json_prod = get_last_resource(YdoMergedRubricsSmallDump, 'stable')
        rubrics_json_testing = get_last_resource(YdoMergedRubricsSmallDump, 'testing')
        features_json_prod = get_last_resource(YdoFeaturesJson, 'stable')
        features_json_testing = get_last_resource(YdoFeaturesJson, 'testing')

        self.Context.order_customizer_json_prod_id = order_customizer_production_resource.id
        self.Context.order_customizer_json_testing_id = order_customizer_testing_resource.id
        self.Context.rubrics_json_prod_id = rubrics_json_prod.id
        self.Context.rubrics_json_testing_id = rubrics_json_testing.id
        self.Context.features_json_prod_id = features_json_prod.id
        self.Context.features_json_testing_id = features_json_testing.id

        cmd_params = [
            self.Parameters.ydo_order_clarifier_config_builder_path,
            '--base-config-raw',
            str(sdk2.ResourceData(self.Parameters.base_config_raw_resource).path),
            '--input-config-raw',
            'input_order_customizer_raw.json',
            '--last-config-raw',
            str(sdk2.ResourceData(get_last_resource(YdoRawOrderCustomizerJson, 'testing_or_stable')).path),
            '--rubrics-json-testing',
            str(sdk2.ResourceData(rubrics_json_testing).path),
            '--features-json-testing',
            str(sdk2.ResourceData(features_json_testing).path),
            '--rubrics-json-production',
            str(sdk2.ResourceData(rubrics_json_prod).path),
            '--features-json-production',
            str(sdk2.ResourceData(features_json_prod).path),
            '--output-config-raw',
            str(order_customizer_output_raw_resource_data.path),
            '--output-config-testing',
            str(order_customizer_testing_resource_data.path),
            '--output-config-production',
            str(order_customizer_production_resource_data.path),
        ]
        if self.Parameters.input_config_as_diff:
            cmd_params.append('--input_config_as_diff')
        if self.Parameters.additional_cmd_params:
            cmd_params.extend(self.Parameters.additional_cmd_params.split())

        execute_cmd(cmd_params, 'release_order_customizer_json')

        try:
            with open('merge_conflict_dump', 'r') as f:
                self.Parameters.merge_conflict_dump = f.read()
        except IOError:
            pass

        try:
            with open('validation_error_dump_testing', 'r') as f:
                self.Parameters.validation_error_dump_testing = f.read()
            raise common.errors.TaskFailure("Validation errors. Check output parameters")
        except IOError:
            order_customizer_testing_resource_data.ready()

        try:
            with open('validation_error_dump_production', 'r') as f:
                self.Parameters.validation_error_dump_production = f.read()
            raise common.errors.TaskFailure("Validation errors. Check output parameters")
        except IOError:
            order_customizer_production_resource_data.ready()

        order_customizer_output_raw_resource_data.ready()

    def _is_testing(self, release_parameters):
        return release_parameters.get('release_status') == 'testing'

    def _is_stable(self, release_parameters):
        return release_parameters.get('release_status') == 'stable'

    def _build_form_answers_gzt(self, testing=True):
        if testing:
            order_customizer_json_testing = sdk2.Resource[self.Context.order_customizer_json_testing_id]
            order_customizer_json_prod = None
            rubricator = sdk2.Resource[self.Context.rubrics_json_testing_id]
            features_json = sdk2.Resource[self.Context.features_json_testing_id]
        else:
            order_customizer_json_testing = None
            order_customizer_json_prod = sdk2.Resource[self.Context.order_customizer_json_prod_id]
            rubricator = sdk2.Resource[self.Context.rubrics_json_prod_id]
            features_json = sdk2.Resource[self.Context.features_json_prod_id]

        build_task = YdoReleaseFormAnswersGzt(
            self,
            self_release_parameter=None,
            description='Build and release ydo_form_answers.gzt.bin',
            release_chooser='testing_or_stable' if testing else 'stable',
            order_customizer_json_testing=order_customizer_json_testing,
            order_customizer_json_prod=order_customizer_json_prod,
            rubricator=rubricator,
            features=features_json,
        )
        build_task.save()

        if testing:
            self.Context.form_answers_gzt_testing_task_id = build_task.id
        else:
            self.Context.form_answers_gzt_prod_task_id = build_task.id
        build_task.enqueue()

    def _release_form_answers_gzt(self, parameters):
        if self._is_testing(parameters):
            task_to_release = self.Context.form_answers_gzt_testing_task_id
            release_type = 'testing'
        elif self._is_stable(parameters):
            task_to_release = self.Context.form_answers_gzt_prod_task_id
            release_type = 'stable'
        else:
            return

        if self.Parameters.build_and_release_form_answers_gzt:
            YdoReleaseTask(
                self,
                task_to_release=sdk2.Task[task_to_release],
                release_type=release_type,
            ).enqueue()

    def _build_order_customizer_additionals(self, testing=True):
        if testing:
            order_customizer_json_testing = sdk2.Resource[self.Context.order_customizer_json_testing_id]
            order_customizer_json_prod = None
        else:
            order_customizer_json_testing = None
            order_customizer_json_prod = sdk2.Resource[self.Context.order_customizer_json_prod_id]

        build_task = YdoReleaseOrderCustomizerAdditionals(
            self,
            self_release_parameter=None,
            description='Build and release ydo_order_customizer_seo_data.json',
            order_customizer_json_testing=order_customizer_json_testing,
            order_customizer_json_prod=order_customizer_json_prod,
            release_chooser='testing_or_stable' if testing else 'stable',
            secret_env={'YDB_TOKEN': 'ydb-token', 'SOLOMON_TOKEN': 'solomon-token'},
            use_custom_config=False,
            arcadia_config='testing.yaml' if testing else 'production.yaml',
        )
        build_task.save()

        if testing:
            self.Context.order_customizer_additionals_testing_task_id = build_task.id
        else:
            self.Context.order_customizer_additionals_prod_task_id = build_task.id
        build_task.enqueue()

    def _release_order_customizer_additionals(self, parameters):
        if self._is_testing(parameters):
            task_to_release = self.Context.order_customizer_additionals_testing_task_id
            release_type = 'testing'
        elif self._is_stable(parameters):
            task_to_release = self.Context.order_customizer_additionals_prod_task_id
            release_type = 'stable'
        else:
            return

        if self.Parameters.build_and_release_order_customizer_additionals:
            YdoReleaseTask(
                self,
                task_to_release=sdk2.Task[task_to_release],
                release_type=release_type,
            ).enqueue()

    def on_release(self, parameters):
        nanny.ReleaseToNannyTask2.on_release(self, parameters)
        sdk2.Task.on_release(self, parameters)

        if self._is_testing(parameters) or self._is_stable(parameters):
            self._release_form_answers_gzt(parameters)
            self._release_order_customizer_additionals(parameters)
            for resource in YdoRawOrderCustomizerJson.find(task=self).limit(0):
                resource.released = 'testing'

    def on_success(self, prev_status):
        if self.Parameters.build_and_release_form_answers_gzt:
            self._build_form_answers_gzt(testing=True)
            self._build_form_answers_gzt(testing=False)

        if self.Parameters.build_and_release_order_customizer_additionals:
            self._build_order_customizer_additionals(testing=True)
            self._build_order_customizer_additionals(testing=False)

        SelfReleaseOnSuccessMixin.on_success(self, prev_status)
        sdk2.Task.on_success(self, prev_status)


def get_last_resource(resource_type, release_status):
    if release_status == 'testing_or_stable':
        stable = resource_type.find(attrs=dict(released='stable')).first()
        testing = resource_type.find(attrs=dict(released='testing')).first()
        if testing is None:
            return stable
        if stable is None:
            return testing
        return max(stable, testing, key=lambda resource: resource.id)

    last_resource = resource_type.find(attrs=dict(released=release_status)).first()
    if last_resource is not None:
        return last_resource
    return None
