from sandbox import sdk2, common
from sandbox.projects.common.nanny import nanny
import sandbox.common.types.task as ctt

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoOrderCustomizerJson,
    YdoTestingOrderCustomizerJson,
    YdoFeaturesJson,
)
from sandbox.projects.ydo.solomon_mixin import SolomonByTaskTypeMixin
from sandbox.projects.ydo.parameters_view import generate_view_for_yt
from sandbox.projects.ydo.resource_types import (
    YdoFormAnswersGzt,
    YdoFormAnswersGztCompilerExecutable,
    YdoFormAnswersSynonymsGzt
)
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.ydo.self_release.ReleaseParentTask import (
    SelfReleaseOnSuccessMixin,
    SelfReleaseOnSuccessMixinParameters
)


class YdoFormAnswersGztBuilderExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoFormAnswersGztRaw(sdk2.Resource):
    releasable = False


class YdoReleaseFormAnswersGzt(sdk2.Task, SelfReleaseOnSuccessMixin, nanny.ReleaseToNannyTask2, SolomonByTaskTypeMixin):
    class Parameters(generate_view_for_yt(
        ydo_form_answers_gzt_builder=YdoFormAnswersGztBuilderExecutable,
        ydo_form_answers_gzt_compiler=YdoFormAnswersGztCompilerExecutable,
        ydo_form_answers_synonyms_gzt=YdoFormAnswersSynonymsGzt,
        with_environ=True,
    )):
        self_release_parameters = SelfReleaseOnSuccessMixinParameters()

        builder_additional_cmd_params = sdk2.parameters.String('Additional gzt builder cmd args')
        compiler_additional_cmd_params = sdk2.parameters.String('Additional gzt compiler cmd args')

        order_customizer_json_testing = sdk2.parameters.Resource(
            'order_customizer.json testing',
            resource_type=YdoTestingOrderCustomizerJson,
            required=False,
        )
        order_customizer_json_prod = sdk2.parameters.Resource(
            'order_customizer.json prod',
            resource_type=YdoOrderCustomizerJson,
            required=False,
        )
        rubricator = sdk2.parameters.Resource(
            'rubrics.json',
            resource_type=YdoMergedRubricsSmallDump,
            required=True,
        )
        features = sdk2.parameters.Resource(
            'features.json',
            resource_type=YdoFeaturesJson,
            required=True,
        )

    class Context(sdk2.Task.Context):
        is_testing = False

    class Requirements(sdk2.Requirements):
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='release_form_answers_gzt_queue')
            ]
        )
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def _check_input_resources(self):
        has_prod_customizer = self.Parameters.order_customizer_json_prod is not None
        has_testing_customizer = self.Parameters.order_customizer_json_testing is not None

        assert has_prod_customizer or has_testing_customizer, "Either one of passed order_customizer resources must be specified!"
        assert not (has_prod_customizer and has_testing_customizer), "Only one of passed order_customizer resources must be specified"

    def on_execute(self):
        self._check_input_resources()

        order_customizer_json = self.Parameters.order_customizer_json_prod or self.Parameters.order_customizer_json_testing
        order_customizer_json_path = str(sdk2.ResourceData(order_customizer_json).path)
        self.Context.is_testing = self.Parameters.order_customizer_json_prod is None

        synonyms_path = self.Parameters.ydo_form_answers_synonyms_gzt_path
        rubrics_json_path = str(sdk2.ResourceData(self.Parameters.rubricator).path)
        features_json_path = str(sdk2.ResourceData(self.Parameters.features).path)

        ydo_form_answers_gzt_resource = YdoFormAnswersGzt(self, 'ydo_form_answers.gzt.bin', 'ydo_form_answers.gzt.bin')
        ydo_form_answers_gzt_raw_resource = YdoFormAnswersGztRaw(self, 'ydo_form_answers.gzt', 'ydo_form_answers.gzt')

        builder_cmd_params = [
            self.Parameters.ydo_form_answers_gzt_builder_path,
            '-g', order_customizer_json_path,
            '-r', rubrics_json_path,
            '--synonyms', synonyms_path,
            '--features', features_json_path,
            '--answers-gzt', str(ydo_form_answers_gzt_raw_resource.path),
            '--debug',
        ]
        if self.Parameters.builder_additional_cmd_params:
            builder_cmd_params.extend(self.Parameters.builder_additional_cmd_params.split())

        execute_cmd(builder_cmd_params, 'Build gazeteers', env=self.Parameters.get_environ())

        # TODO: add parameters for proto path and arcadia revision
        sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url('ydo/libs/order_customizer/proto'), path='proto')
        compiler_cmd_params = [
            self.Parameters.ydo_form_answers_gzt_compiler_path,
            str(ydo_form_answers_gzt_raw_resource.path),
            str(ydo_form_answers_gzt_resource.path),
            '-I', 'proto'
        ]
        if self.Parameters.compiler_additional_cmd_params:
            compiler_cmd_params.extend(self.Parameters.compiler_additional_cmd_params.split())
        execute_cmd(compiler_cmd_params, 'Compile gazeteers')

    def on_release(self, parameters):
        if "release_status" in parameters:
            is_testing_release_target = parameters["release_status"] not in ["stable", "prestable"]
            assert self.Context.is_testing == is_testing_release_target, "Cannot release testing gzt to stable"

        nanny.ReleaseToNannyTask2.on_release(self, parameters)
        sdk2.Task.on_release(self, parameters)

    def on_success(self, prev_status):
        SelfReleaseOnSuccessMixin.on_success(self, prev_status)
        sdk2.Task.on_success(self, prev_status)

    def on_break(self, *args, **kwargs):
        self.Parameters.save_info_about_resources()
        SolomonByTaskTypeMixin.on_break(self, *args, **kwargs)

    def on_finish(self, *args, **kwargs):
        self.Parameters.save_info_about_resources()
        SolomonByTaskTypeMixin.on_finish(self, *args, **kwargs)
