import logging
import os

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo import (
    execute_cmd,
    get_now_utc,
    ydo_releasers,
    YdoFeaturesJson,
)
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables
from sandbox.projects.ydo.ferryman import (
    start_ferryman_process,
    wait_ferryman,
)
from sandbox.projects.ydo.parameters_view import generate_view_for_yt
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE


class YdoTeleportIndexerExecutable(sdk2.Resource):
    """
        Generate table with proto documents for SAAS
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoTeleportIndexer(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment("yandex-yt"),
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(
        generate_view_for_yt(
            indexer=YdoTeleportIndexerExecutable,
            rubrics_json=YdoMergedRubricsSmallDump,
            features_json=YdoFeaturesJson,
            geobase_file=GEODATA6BIN_STABLE,
            with_environ=True,
        )
    ):
        ferryman_name = sdk2.parameters.String("Ferryman name", required=True)
        teleport_saas_namespace = sdk2.parameters.String("SAAS namespace (kps) for teleport", required=True)
        degrade_level = sdk2.parameters.Float("Max allowed difference between tables row counts", default=1.0)

        res_dir = sdk2.parameters.String("Index resulting directory", required=True)
        ydb_backup_path = sdk2.parameters.String("Path to ydb backup", required=True)

    def create_index(self):
        self.Context.timestamp = get_now_utc()

        def table_name(template, kps):
            return os.path.join(self.Parameters.res_dir, template.format(ts=self.Context.timestamp, kps=kps))

        self.Context.out_teleport_table = table_name("{ts}_teleport_kps={kps}", self.Parameters.teleport_saas_namespace)

        logging.info("Indexing...")

        cmd = [
            self.Parameters.indexer_path,
            "--ydb_backup_path", self.Context.out_org_permalink_table,
            "--geobase_file", self.Parameters.geobase_file_path,
            "--rubrics_json", self.Parameters.rubrics_json_path,
            "--features_json", self.Parameters.features_json_path,
            "--teleport_res_table", self.Context.out_teleport_table,
        ]

        execute_cmd(cmd, "ydo_teleport_indexer", "Failed to create teleport index")

    def link(self):
        tables_to_link = {
            self.Context.out_teleport_table: os.path.join(self.Parameters.res_dir, "current_teleport_index"),
        }
        link_task = YdoBackupLinkTables(
            self,
            description="Link tables for task {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.yt_host,
            yt_vault_token="yt-token",
            yt_tables=tables_to_link,
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def on_execute(self):
        self.yt_host = "hahn"

        with self.memoize_stage.firststage:
            self.create_index()

            ferryman_args = {
                "ferryman_name": self.Parameters.ferryman_name,
                "timestamp": self.Context.timestamp,
                "yt_token": sdk2.Vault.data(self.owner, "yt-token"),
                "degrade_level": self.Parameters.degrade_level,
                "cluster": self.yt_host,
                "proto": True,
            }
            out_tables2saas_namespaces = {
                self.Context.out_teleport_table: self.Parameters.teleport_saas_namespace,
            }
            self.Context.ferryman_batch_id = start_ferryman_process(out_tables2saas_namespaces, **ferryman_args)

        with self.memoize_stage.wait_ferryman(9):
            wait_ferryman(self.Parameters.ferryman_name, self.Context.ferryman_batch_id)

        with self.memoize_stage.link_to_current:
            self.link()

        with self.memoize_stage.clean_folder:
            clean_folder(self, self.Parameters.res_dir, yt_host=self.yt_host, history_size=4)

        logging.info("Done")
