import os

import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo import ydo_releasers

from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables
from sandbox.projects.ydo.parameters_view import generate_view


class YdoCalcOrderQualityFactorsDataExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoCalcOrderQualityFactorsDataTask(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(generate_view(
        calc_order_quality_factors_data_executable=YdoCalcOrderQualityFactorsDataExecutable, check_owner=True, with_environ=True)
    ):
        customer_docs_backup_path = sdk2.parameters.String(
            "Path to directory where to store table with SAAS customer docs",
            required=True
        )
        customer_states_backup_path = sdk2.parameters.String(
            "Path to directory where to store table with customer states",
            required=True
        )
        specialization_docs_backup_path = sdk2.parameters.String(
            "Path to directory where to store table with SAAS specialization docs",
            required=True
        )
        specialization_states_backup_path = sdk2.parameters.String(
            "Path to directory where to store table with specialization states",
            required=True
        )
        db_dump_path = sdk2.parameters.String(
            "Path to directory with db dump",
            required=True
        )
        bad_reacted_workers_path = sdk2.parameters.String(
            "Path to yt table with bad reacted workers",
            required=True
        )
        verified_workers_path = sdk2.parameters.String(
            "Path to yt table with verified workers and their verified rating",
            required=True
        )
        factors_data_service_name = sdk2.parameters.String(
            "SAAS service name",
            required=True
        )
        factors_data_ctype = sdk2.parameters.String(
            "SAAS service ctype",
            required=True
        )
        logbroker_host = sdk2.parameters.String(
            "Logbroker host",
            required=True
        )
        self_tvm_id = sdk2.parameters.String(
            "Task tvm id",
            required=True
        )
        logbroker_tvm_id = sdk2.parameters.String(
            "Logbroker tvm id",
            required=True
        )
        rate_limit = sdk2.parameters.String(
            "Limit on pushed docs per second",
            required=True
        )

    def link(self, paths):
        import yt.wrapper as yt

        env = self.Parameters.get_environ()
        yt.config['token'] = env['YT_TOKEN']
        yt.config['proxy']['url'] = 'hahn'

        tables_to_link = {max(filter(lambda p: not p.endswith('current'), yt.list(path, absolute=True))): os.path.join(path, 'current') for path in paths}
        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host='hahn',
            yt_vault_token='yt-token',
            yt_tables=tables_to_link
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.run_cmd:
            cmd = [
                self.Parameters.calc_order_quality_factors_data_executable_path,
                '--customer_docs_backup', self.Parameters.customer_docs_backup_path,
                '--customer_states_backup', self.Parameters.customer_states_backup_path,
                '--specialization_docs_backup', self.Parameters.specialization_docs_backup_path,
                '--specialization_states_backup', self.Parameters.specialization_states_backup_path,
                '--dump_from_yt', self.Parameters.db_dump_path,
                '--bad_reacted_workers', self.Parameters.bad_reacted_workers_path,
                '--verified_workers', self.Parameters.verified_workers_path,
                '--factors_data_service_name', self.Parameters.factors_data_service_name,
                '--factors_data_ctype', self.Parameters.factors_data_ctype,
                '--logbroker_host', self.Parameters.logbroker_host,
                '--self_tvm_id', self.Parameters.self_tvm_id,
                '--logbroker_tvm_id', self.Parameters.logbroker_tvm_id,
                '--rate_limit', self.Parameters.rate_limit,
            ]
            execute_cmd(cmd, "calc_order_quality_factors_data", "calc_order_quality_factors_data failed", env=self.Parameters.get_environ())

        with self.memoize_stage.clean_docs_folder:
            clean_folder(self, self.Parameters.customer_docs_backup_path)

        with self.memoize_stage.clean_docs_folder:
            clean_folder(self, self.Parameters.customer_states_backup_path)

        with self.memoize_stage.clean_states_folder:
            clean_folder(self, self.Parameters.specialization_docs_backup_path)

        with self.memoize_stage.clean_states_folder:
            clean_folder(self, self.Parameters.specialization_states_backup_path)

        with self.memoize_stage.link_to_current_docs:
            self.link((self.Parameters.customer_docs_backup_path,
                       self.Parameters.customer_states_backup_path,
                       self.Parameters.specialization_docs_backup_path,
                       self.Parameters.specialization_states_backup_path))
