# coding: utf-8
import logging

import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common import task_env
from sandbox.projects.ydo import ydo_releasers
from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class YdoOrderWeightUpdaterExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoOrderWeightUpdater(sdk2.Task):
    """
    Order's weight updating. set only 0 for zero-order
    YDO-32427
    """

    class Requirements(task_env.TinyRequirements):
        ram = 1024
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(
        generate_view(ydo_order_weight_updater=YdoOrderWeightUpdaterExecutable, check_owner=True, with_environ=True)
    ):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            service_complaints_table = sdk2.parameters.String('Path to service_complaints backup', required=True)
            service_incidents_table = sdk2.parameters.String('Path to service_incidents backup', required=True)

        calculation_window = sdk2.parameters.Integer('Calculation window', default=24 * 60 * 60 * 60)
        logging_config = sdk2.parameters.String('LoggingConfig')

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        cmd = [self.Parameters.ydo_order_weight_updater_path]

        self.add_param_to_cmd(cmd, '--kikimr-host', self.Parameters.ydb_host)
        self.add_param_to_cmd(cmd, '--kikimr-home', self.Parameters.ydb_home)
        self.add_param_to_cmd(cmd, '--kikimr-database', self.Parameters.ydb_database)
        self.add_param_to_cmd(cmd, '--kikimr-port', str(self.Parameters.ydb_port))
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))

        self.add_param_to_cmd(cmd, '--service_complaints', self.Parameters.service_complaints_table)
        self.add_param_to_cmd(cmd, '--service_incidents', self.Parameters.service_incidents_table)
        self.add_param_to_cmd(cmd, '--calculation_window', str(self.Parameters.calculation_window))
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)

        execute_cmd(
            cmd, 'ydo_order_weight_updater', 'ydo_order_weight_updater failed', env=self.Parameters.get_environ()
        )
