import logging
import os
import tempfile
import yaml

import sandbox.common.errors as ce
import sandbox.common.types.misc as ctm
import sandbox.sdk2 as sdk2

from sandbox.projects.common import file_utils as fu

import sandbox.projects.common.arcadia.sdk as arcadia_sdk


def resource_id_attr_name(name):
    return name + '_resource_id'


def resource_attr_name(name):
    return name + '_resource'


def path_attr_name(name):
    return name + '_path'


def format_context_field(resource_name):
    return 'cached_resource_id_for_{}'.format(resource_name)


def get_resource_id_from_cache(resource_name):
    task_context = sdk2.Task.current.Context
    resource_id = getattr(task_context, format_context_field(resource_name))
    if resource_id is ctm.NotExists:
        return None
    return resource_id


def save_resource_id_in_cache(resource_name, resource_id):
    task_context = sdk2.Task.current.Context
    setattr(task_context, format_context_field(resource_name), resource_id)
    if task_context.cached_resources is ctm.NotExists:
        task_context.cached_resources = {}
    task_context.cached_resources[resource_name] = resource_id


def generate_resource_finder(name, resource_type, check_owner):
    def resource_finder(self):
        owner_kwargs = dict() if not check_owner else dict(owner=self.owner)
        if self.release_chooser == 'stable':
            return resource_type.find(attrs=dict(released='stable'), **owner_kwargs).first()
        elif self.release_chooser == 'testing_or_stable':
            stable = resource_type.find(attrs=dict(released='stable'), **owner_kwargs).first()
            testing = resource_type.find(attrs=dict(released='testing'), **owner_kwargs).first()
            if testing is None:
                return stable
            if stable is None:
                return testing
            return max(stable, testing, key=lambda resource: resource.id)
        else:
            assert self.release_chooser == 'custom'
            return getattr(self, resource_id_attr_name(name))

    def cached_resource_finder(self):
        if get_resource_id_from_cache(name) is None:
            resource = resource_finder(self)
            save_resource_id_in_cache(name, resource.id)

        return resource_type.find(id=get_resource_id_from_cache(name)).first()

    return cached_resource_finder


def generate_resource_data_downloader(name):
    def resource_data_uploader(self):
        resource = getattr(self, resource_attr_name(name))
        resource_data = sdk2.ResourceData(resource)
        return str(resource_data.path)

    return resource_data_uploader


def get_env(self):
    env = os.environ.copy()
    for env_name, value in self.env.iteritems():
        env[env_name] = str(value)

    if not hasattr(self, 'yav_secret_env'):
        for env_name, vault_name in self.secret_env.iteritems():
            env[env_name] = sdk2.Vault.data(self.owner, vault_name)
    else:
        for env_name, yav_vault_value in self.yav_secret_env.iteritems():
            secret = sdk2.yav.Secret.__decode__(yav_vault_value)
            env[env_name] = secret.value()

    return env


def save_info_about_resources(self):
    task = sdk2.Task.current
    if task.Context.cached_resources is not ctm.NotExists:
        resource_list_str = ''.join(
            '<li><a href="https://sandbox.yandex-team.ru/resource/{}/view">{}</a></li>'.format(resource_id, name)
            for name, resource_id in task.Context.cached_resources.items()
        )
        task.set_info('Loaded resources:<ul>{}</ul>'.format(resource_list_str), do_escape=False)


class ResourceWrapper(object):
    def __init__(self, resource_class, check_owner=False):
        self.resource_class = resource_class
        self.check_owner = check_owner


def generate_view(**kwargs):
    check_owner = kwargs.pop('check_owner', False)
    with_environ = kwargs.pop('with_environ', False)
    default_env_vars = kwargs.pop('default_env_vars', {})
    default_secret_env_vars = kwargs.pop('default_secret_env_vars', {})

    use_yav_secrets = kwargs.pop('use_yav_secrets', False)

    custom_subfields = list()
    dct_kwargs = dict()
    for name, resource_type in kwargs.iteritems():
        resource_check_owner = check_owner
        unwrapped_resource_type = resource_type
        if isinstance(resource_type, ResourceWrapper):
            unwrapped_resource_type = resource_type.resource_class
            resource_check_owner = resource_type.check_owner

        attr_name = resource_id_attr_name(name)
        custom_subfields.append(attr_name)
        dct_kwargs[attr_name] = sdk2.parameters.Resource(name, resource_type=unwrapped_resource_type, required=True)
        dct_kwargs[resource_attr_name(name)] = property(
            generate_resource_finder(name, unwrapped_resource_type, resource_check_owner)
        )
        dct_kwargs[path_attr_name(name)] = property(generate_resource_data_downloader(name))

    dct = dict(
        release_chooser=sdk2.parameters.RadioGroup(
            'Choose release type',
            choices=[
                ('stable', 'stable'),
                ('testing_or_stable', 'testing_or_stable'),
                ('custom', 'custom'),
            ],
            default='stable',
            sub_fields=dict(custom=custom_subfields),
        ),
        **dct_kwargs
    )
    dct['__names__'] = ['release_chooser'] + custom_subfields
    if with_environ:
        dct['env'] = sdk2.parameters.Dict(
            'Environment variables',
            default=default_env_vars,
        )
        if use_yav_secrets:
            dct['yav_secret_env'] = sdk2.parameters.Dict(
                'Secret environment variables',
                default=default_secret_env_vars,
            )
            dct['__names__'].append('yav_secret_env')
        else:
            dct['secret_env'] = sdk2.parameters.Dict('Secret environment variables', default=default_secret_env_vars)
            dct['__names__'].append('secret_env')

        dct['get_environ'] = get_env
        dct['__names__'].append('env')

    dct['save_info_about_resources'] = save_info_about_resources

    dct['__names__'] = tuple(dct['__names__'])
    return type('ParamsView', (sdk2.Parameters,), dct)


def generate_view_for_yt(**kwargs):
    default_env_vars = kwargs.pop('default_env_vars', {})
    default_env_vars['YT_PROXY'] = 'hahn.yt.yandex.net'
    default_secret_env_vars = kwargs.pop('default_secret_env_vars', {})
    default_secret_env_vars['YT_TOKEN'] = 'yt-token'
    return generate_view(default_env_vars=default_env_vars, default_secret_env_vars=default_secret_env_vars, **kwargs)


def with_config(parent_parameters, configs_folder, configs_choices, auto_post_process=False):
    class ConfigParameters(parent_parameters):
        use_custom_config = sdk2.parameters.Bool('Custom config', default=False)
        with use_custom_config.value[True]:
            custom_config = sdk2.parameters.String('Config', multiline=True)
        with use_custom_config.value[False]:
            arcadia_config = sdk2.parameters.RadioGroup(
                'Config',
                choices=configs_choices,
            )
            with sdk2.parameters.Output:
                config = sdk2.parameters.String('Config text', multiline=True)

        @property
        def config_path(self):
            temp_config = tempfile.NamedTemporaryFile().name
            if self.use_custom_config:
                config_content = self.custom_config
            else:
                arc_token = None
                try:
                    arc_token = sdk2.Vault.data(self.owner, 'ARC_TOKEN')
                except ce.VaultNotFound:
                    pass
                additional_mount_kwargs = {}
                if arc_token:
                    additional_mount_kwargs = dict(
                        use_arc_instead_of_aapi=True,
                        arc_oauth_token=arc_token,
                    )
                    logging.info('Use arc instead of aapi')
                with arcadia_sdk.mount_arc_path(
                    os.path.join('arcadia:/arc/trunk/arcadia/', configs_folder), **additional_mount_kwargs
                ) as configs_path:
                    config_path = open(os.path.join(configs_path, self.arcadia_config))
                    with config_path as config:
                        config_content = config.read()
                        self.config = config_content

            if auto_post_process:
                config_content = self.auto_post_process_config(config_content)
                logging.debug('Auto post processed config:\n%s', config_content)
            else:
                logging.debug('No need to process config fields')

            fu.write_file(temp_config, config_content)
            return temp_config

        def auto_post_process_config(self, config_content):
            config = yaml.load(config_content)
            config = self.auto_process_fields(config, [])
            return yaml.dump(config, default_flow_style=False)

        def auto_process_fields(self, config, current_path):
            if isinstance(config, list):
                for i, obj in enumerate(config):
                    config[i] = self.auto_process_fields(obj, current_path + [str(i)])

            elif isinstance(config, dict):
                if config.get('type') != 'sandbox_resource':
                    for k, obj in config.items():
                        config[k] = self.auto_process_fields(obj, current_path + [k])
                else:
                    cache_key = '/'.join(current_path)
                    if get_resource_id_from_cache(cache_key) is None:
                        if 'id' in config:
                            resource_id = int(config['id'])
                        else:
                            assert 'resource_type' in config
                            resource_type = config['resource_type']
                            owner_kwargs = dict() if config.get('dont_check_owner') else dict(owner=self.owner)
                            if self.release_chooser == 'stable':
                                resource = sdk2.Resource.find(
                                    resource_type=resource_type, attrs=dict(released='stable'), **owner_kwargs
                                ).first()
                            elif self.release_chooser == 'testing_or_stable':
                                stable = sdk2.Resource.find(
                                    resource_type=resource_type, attrs=dict(released='stable'), **owner_kwargs
                                ).first()
                                testing = sdk2.Resource.find(
                                    resource_type=resource_type, attrs=dict(released='testing'), **owner_kwargs
                                ).first()
                                if testing is None:
                                    resource = stable
                                elif stable is None:
                                    resource = testing
                                else:
                                    resource = max(stable, testing, key=lambda resource: resource.id)
                            else:
                                assert self.release_chooser == 'custom'
                                raise NotImplemented('use id for specify resource in config in custom mode')
                            resource_id = resource.id
                        save_resource_id_in_cache(cache_key, resource_id)

                    resource_id = get_resource_id_from_cache(cache_key)
                    resource = sdk2.Resource.find(id=resource_id).first()
                    resource_data = sdk2.ResourceData(resource)
                    config = str(resource_data.path)

            return config

    return ConfigParameters
