import logging

from sandbox import sdk2
from sandbox.projects.ydo import ydo_releasers, execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view


class YdoTaxiPassportCheckerExecutable(sdk2.Resource):
    """Fetch answers from Taxi-queue and push new issues"""
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoTaxiPassportChecker(sdk2.Task):
    """Task that fetches check results from Taxi-queue and pushes new data to taxi."""
    class Parameters(generate_view(
        passport_checker=YdoTaxiPassportCheckerExecutable,
        check_owner=True,
        with_environ=True,
        default_env_vars={
            'YT_PROXY': 'hahn.yt.yandex.net',
            'YT_LOG_LEVEL': 'INFO',
        },
    )):
        ydo_tokens = sdk2.parameters.YavSecret("Ydo ydb token", required=True, default='sec-01eejahmvyzqjv50qt9smmq898')
        s3_ydo = sdk2.parameters.YavSecret("Ydo s3 secrets", required=True, default='sec-01dtsgsd0zz2ednshq0gax5mak')

        yt_backup_path = sdk2.parameters.String('Path to YDB backup', required=True)
        yt_service_squeeze_path = sdk2.parameters.String('Path to squeezed search and order clicks dir', required=True)

        priority_delta = sdk2.parameters.Integer('Number of days to take into account for workers prioritization', required=True, default=14)

        ydb_host = sdk2.parameters.String('YDB host', required=True)
        ydb_port = sdk2.parameters.Integer('YDB port', required=True, default=2135)
        ydb_database = sdk2.parameters.String('YDB database', required=True)
        ydb_home = sdk2.parameters.String('Path to YDB home dir', required=True)

        ts_for_skip_red_passports = sdk2.parameters.Integer('Timestamp starting from which to use last red passport to skip the check', required=True, default=1619816400)

        taxi_checker_capacity = sdk2.parameters.Integer('Max amount of passports that can be checked in taxi in one day', required=True, default=20000)
        base_yt_path_to_taxi_check_queues = sdk2.parameters.String('YT path to taxi checks queues', required=True, default='//home/ydo/tools/workers_quality/pass_check')

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def proceed_passports_batch(self):
        env = self.Parameters.get_environ()
        env['YT_TOKEN'] = self.Parameters.ydo_tokens.data()['YT_TOKEN']
        env['YDB_TOKEN'] = self.Parameters.ydo_tokens.data()['YDB_TOKEN']
        env['S3_ACCESS_KEY'] = self.Parameters.s3_ydo.data()['s3_access_key']
        env['S3_SECRET_ACCESS_KEY'] = self.Parameters.s3_ydo.data()['s3_secret_access_key']
        env['SECURE_IMAGES_AES_KEY'] = self.Parameters.s3_ydo.data()['secure_images_aes_key']

        execute_cmd(
            [
                self.Parameters.passport_checker_path,
                '--yt_backup_path', self.Parameters.yt_backup_path,
                '--yt_service_squeeze_path', self.Parameters.yt_service_squeeze_path,
                '--priority_delta', str(self.Parameters.priority_delta),
                '--ydb_host', self.Parameters.ydb_host,
                '--ydb_port', str(self.Parameters.ydb_port),
                '--ydb_database', self.Parameters.ydb_database,
                '--ydb_home', self.Parameters.ydb_home,
                '--ts_for_skip_red_passports', str(self.Parameters.ts_for_skip_red_passports),
                '--taxi_checker_capacity', str(self.Parameters.taxi_checker_capacity),
                '--base_yt_path_to_taxi_check_queues', self.Parameters.base_yt_path_to_taxi_check_queues,
            ],
            'ydo_passport_checker',
            'Failed to proceed passports batch',
            env=env
        )

    def on_execute(self):
        self.proceed_passports_batch()
        logging.info('Done')
