import datetime

from sandbox import sdk2
from sandbox.projects.ydo import execute_cmd, ydo_releasers
from sandbox.projects.ydo.parameters_view import generate_view_for_yt


class YdoSaasPrepWorkerStatsBinary(sdk2.Resource):
    releasers = ydo_releasers
    executable = True
    releasable = True


class YdoPrepareSaasStats(sdk2.Task):
    """ Prepare worker stats from YDO for SaaS. """

    class Parameters(generate_view_for_yt(
        ydo_saas_prep_worker_stats=YdoSaasPrepWorkerStatsBinary,
        check_owner=True,
        with_environ=True
    )):
        input_date = sdk2.parameters.RadioGroup(
            "Input date",
            choices=(
                ('yesterday', 'yesterday'),
                ('custom', 'custom'),
            ),
            required=True,
            default='yesterday',
            sub_fields={'custom': ['custom_input_date']}
        )
        custom_input_date = sdk2.parameters.StrictString(
            'YYYY-MM-DD',
            regexp=r'\d{4}-\d{2}-\d{2}',
            required=True
        )

        input_path = sdk2.parameters.String(
            'Input node path',
            required=True,
            default='//home/ydo/users/zmiecer/worker-stats'
        )

        output_path = sdk2.parameters.String(
            'Output node path',
            required=True,
            default_value='//home/ydo/users/zmiecer/saas-worker-stats'
        )

        expiration_time = sdk2.parameters.Integer(
            'Expiration time of the output table in days',
            required=True,
            default=90
        )

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_input_date(self):
        if self.Parameters.input_date == 'yesterday':
            yesterday = datetime.datetime.today() - datetime.timedelta(days=1)
            input_date = yesterday.strftime('%Y-%m-%d')
        elif self.Parameters.input_date == 'custom':
            input_date = self.Parameters.custom_input_date
        return input_date

    def on_execute(self):
        input_date = self.get_input_date()
        cmd = [
            self.Parameters.ydo_saas_prep_worker_stats_path,
            '--input_date', input_date,
            '--input_path', self.Parameters.input_path,
            '--output_path', self.Parameters.output_path,
            '--expiration_time', str(self.Parameters.expiration_time)
        ]

        execute_cmd(
            cmd=cmd,
            log_name='ydo_prepare_saas_stats',
            error_message='Saas stats preparator failed',
            env=self.Parameters.get_environ()
        )
