import logging

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment


class YdoGenPhoneToRegionMapExecutable(sdk2.Resource):
    """
        Gen phone to region map executable
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoGenPhoneToRegionMap(sdk2.Task):
    """
        Gen phone to region map
    """

    class Parameters(generate_view(
        gen_phone_region_map=YdoGenPhoneToRegionMapExecutable,
        geobase_snapshot=ResourceWrapper(GEODATA6BIN_STABLE),
        tzdata_snapshot=ResourceWrapper(GEODATATZDATA_STABLE),
        check_owner=True,
        with_environ=True,
        default_env_vars={
            'YT_PROXY': 'hahn',
            'REQUESTS_CA_BUNDLE': '/etc/ssl/certs/',
        },
        default_secret_env_vars={
            'TVM_SECRET': 'ydo_tvm_secret_prod',
            'YT_TOKEN': 'yt-token',
        },
    )):
        phones_data_urls = sdk2.parameters.List('Urls to download phones data from')

        phone_to_region_yt_table = sdk2.parameters.String('YT table to store created mapping', required=True)

        region_to_geoid_yt_table = sdk2.parameters.String('YT table to read from or write to region-to-geoid mapping', required=True)

        make_region_to_geoid_map = sdk2.parameters.Bool('Creates region to geoid mapping and saves it instead of using existing one', default=False)

        skip_extending_region_to_geoid = sdk2.parameters.Bool('Will not make requests to geocoder to extend existing region-to-geoid mapping with missing regions', default=False)

        batch_geocoder = sdk2.parameters.Bool('Use batch installation of geocoder instead of production installation', default=False)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def run(self):
        logging.info('Started gen_phone_to_region_map process')
        execute_cmd(
            [
                self.Parameters.gen_phone_region_map_path,
                '--geobase-file', self.Parameters.geobase_snapshot_path,
                '--tz-data', self.Parameters.tzdata_snapshot_path,
                '--download-phones-data',
                '--phone-to-region-yt-table', self.Parameters.phone_to_region_yt_table,
                '--region-to-geoid-yt-table', self.Parameters.region_to_geoid_yt_table,
            ]
            + (['--make-region-to-geoid-map'] if self.Parameters.make_region_to_geoid_map else [])
            + (['--skip-extending-region-to-geoid'] if self.Parameters.skip_extending_region_to_geoid else [])
            + (['--batch-geocoder'] if self.Parameters.batch_geocoder else [])
            + (['--phones-data-urls'] + list(self.Parameters.phones_data_urls)),
            'gen_phone_to_region_map',
            'Error occurred',
            env=self.Parameters.get_environ(),
        )

    def on_execute(self):
        with self.memoize_stage.run:
            self.run()

        logging.info('Done')
