import logging

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo import (
    execute_cmd,
    get_now_utc,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment


class YdoFastCallTrackingExecutable(sdk2.Resource):
    """
        Fast call tracking executable
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoAntispamPhonesExport(sdk2.Resource):
    """
        Antispam phones export yt table dump
    """
    releasable = True
    releasers = ydo_releasers


class YdoFastCallTracking(sdk2.Task):
    """
        Glue calls to connects
    """

    class Parameters(generate_view(
        fast_call_tracking=YdoFastCallTrackingExecutable,
        check_owner=True,
        with_environ=True,
        default_env_vars={
            'REQUESTS_CA_BUNDLE': '/etc/ssl/certs/',
        },
        default_secret_env_vars={
            'YDB_TOKEN': 'ydb-token',
            'TVM_SECRET': 'ydo_tvm_secret_prod',
            'S3_ACCESS_KEY': 'S3_YDO_PHONE_RECORDS_ACCESS_KEY_ID',
            'S3_SECRET_ACCESS_KEY': 'S3_YDO_PHONE_RECORDS_SECRET_ACCESS_KEY',
            'SOLOMON_TOKEN': 'solomon-token',
            'S3_CLOUD_ACCESS_KEY': 'CLOUD_PROXY_PHONES_ACCESS_KEY',
            'S3_CLOUD_SECRET_ACCESS_KEY': 'CLOUD_PROXY_PHONES_SECRET_ACCESS_KEY',
        },
    )):
        ydb_host = sdk2.parameters.String('YDB host', required=True)

        ydb_port = sdk2.parameters.Integer('YDB port', required=True, default=2135)

        ydb_database = sdk2.parameters.String('YDB database', required=True)

        ydb_home = sdk2.parameters.String('Path to YDB home dir', required=True)

        ydb_retries = sdk2.parameters.Integer('YDB retries count', required=True, default=3)

        ydb_retry_delay = sdk2.parameters.Float('YDB delay between retries', required=True, default=3.)

        requests_timeout = sdk2.parameters.Float('Timeout for requests to telephony api', required=True, default=5.)

        tvm_id = sdk2.parameters.Integer('Ydo tvm_id', required=True, default=2002420)

        telephony_env = sdk2.parameters.String('Telephony env', required=True, default='prod')

        joining_period_seconds = sdk2.parameters.Integer('Period in seconds before call we join connects', required=True, default=300)

        previous_info_period_seconds = sdk2.parameters.Integer('Period in seconds before now we take users data from already matched calls', required=True, default=259200)

        thread_pool_workers = sdk2.parameters.Integer('Thread pool workers count', required=True, default=8)

        contacts_delta_ts = sdk2.parameters.Integer('Contacts delta ts', required=True, default=5)

        contacts_deduplication_interval = sdk2.parameters.Integer('Contacts deduplication interval', required=True, default=0)

        delete_proxy_phones_calls_from_db_if_joined = sdk2.parameters.Bool('Delete telephony calls data from db if joined', default=False)

        delete_old_proxy_phones_calls_from_db_interval = sdk2.parameters.Integer('Delete not joined telephony calls data from db interval', required=True, default=1800)

        create_new_contacts_if_missing = sdk2.parameters.Bool('Create new contacts if missing', default=False)

        s3_url = sdk2.parameters.String('Url to MDS', default='http://s3.mds.yandex.net')

        download_call_records = sdk2.parameters.Bool('Download call records and put them to MDS', default=False)

        put_call_record_to_mds_timeout = sdk2.parameters.Float('Timeout to put call record to MDS', default=5.)

        s3_cloud_url = sdk2.parameters.String('Url to YC', default='https://storage.yandexcloud.net')

        upload_record_to_yc = sdk2.parameters.Bool('Download call records and put them to YC', default=False)

        ffmpeg_binary = sdk2.parameters.Resource('ffmpeg binary', required=False)

        put_call_record_to_yc_timeout = sdk2.parameters.Float('Timeout to put call record to YC', default=5.)

        enable_solomon = sdk2.parameters.Bool('Enable solomon', default=False)

        solomon_project = sdk2.parameters.String('Solomon project', default='ydo')

        solomon_cluster = sdk2.parameters.String('Solomon cluster', default='fast_call_tracking')

        solomon_service = sdk2.parameters.String('Solomon service', default='simple_monitoring')

        solomon_label = sdk2.parameters.String('Solomon label', default='debug')

    class Requirements(sdk2.Requirements):
        cores = 8
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def run(self):
        logging.info('Started fast_call_tracking process')
        execute_cmd(
            [
                self.Parameters.fast_call_tracking_path,
                '--ydb_host', self.Parameters.ydb_host,
                '--ydb_port', str(self.Parameters.ydb_port),
                '--ydb_database', self.Parameters.ydb_database,
                '--ydb_home', self.Parameters.ydb_home,
                '--ydb_retries', str(self.Parameters.ydb_retries),
                '--ydb_retry_delay', str(self.Parameters.ydb_retry_delay),
                '--requests_timeout', str(self.Parameters.requests_timeout),
                '--tvm_id', str(self.Parameters.tvm_id),
                '--telephony_env', self.Parameters.telephony_env,
                '--joining_period_seconds', str(self.Parameters.joining_period_seconds),
                '--previous_info_period_seconds', str(self.Parameters.previous_info_period_seconds),
                '--thread_pool_workers', str(self.Parameters.thread_pool_workers),
                '--contacts_delta_ts', str(self.Parameters.contacts_delta_ts),
                '--contacts_deduplication_interval', str(self.Parameters.contacts_deduplication_interval),
                '--delete_old_proxy_phones_calls_from_db_interval', str(self.Parameters.delete_old_proxy_phones_calls_from_db_interval),
            ]
            + (['--delete_proxy_phones_calls_from_db_if_joined'] if self.Parameters.delete_proxy_phones_calls_from_db_if_joined else [])
            + (['--create_new_contacts_if_missing'] if self.Parameters.create_new_contacts_if_missing else [])
            + (
                [
                    '--download_call_records',
                    '--s3_url', self.Parameters.s3_url,
                    '--put_call_record_to_mds_timeout', str(self.Parameters.put_call_record_to_mds_timeout),
                ] if self.Parameters.download_call_records else []
            )
            + (
                [
                    '--upload_record_to_yc',
                    '--ffmpeg_path', str(sdk2.ResourceData(self.Parameters.ffmpeg_binary).path),
                    '--s3_cloud_url', self.Parameters.s3_cloud_url,
                    '--put_call_record_to_yc_timeout', str(self.Parameters.put_call_record_to_yc_timeout),
                ] if self.Parameters.upload_record_to_yc else []
            )
            + (
                [
                    '--enable_solomon',
                    '--solomon_project', self.Parameters.solomon_project,
                    '--solomon_cluster', self.Parameters.solomon_cluster,
                    '--solomon_service', self.Parameters.solomon_service,
                    '--solomon_label', self.Parameters.solomon_label,
                ] if self.Parameters.enable_solomon else []
            ),
            'ydo_fast_call_tracking',
            'Error occurred',
            env=self.Parameters.get_environ(),
        )

    def on_execute(self):
        with self.memoize_stage.run:
            self.run()

        logging.info('Done')
