import datetime
import logging
import os

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox import sdk2

from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
)

from sandbox.projects.yql.RunYQL2 import RunYQL2


# Resource types
class YdoGetPhonesLogsExecutable(sdk2.Resource):
    """
        Aggregate proxy phones calls logs
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


def date_to_str(d):
    return d.strftime('%Y%m%d')


class YdoGetProxyPhonesLogs(sdk2.Task):
    '''Task that aggregates all phone calls logs'''

    class Parameters(generate_view(
        default_secret_env_vars={
            'YDB_TOKEN': 'ydb-token'
        },
        with_environ=True,
        get_phones_logs_executable=YdoGetPhonesLogsExecutable
    )):
        output_dir = sdk2.parameters.String('Path to dir where to store results', required=True)
        start_date = sdk2.parameters.String('Start of period (YYYYMMDD). Yesterday by default.')
        process_today_date = sdk2.parameters.Bool('Process current day')
        with_postprocess = sdk2.parameters.Bool('Run postprocess', default=True)
        ydb_host = sdk2.parameters.String('YDB host', required=True, default='ydb-ru.yandex.net')
        ydb_port = sdk2.parameters.Integer('YDB port', required=True, default=2135)
        ydb_database = sdk2.parameters.String('YDB database', required=True, default='/ru/services/prod/services')
        ydb_home = sdk2.parameters.String('Path to YDB home dir', required=True, default='/ru/services/prod/services/runtime')

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        with self.memoize_stage.ask_api:
            yt_token = sdk2.Vault.data(self.owner, 'yt-token')

            env = self.Parameters.get_environ()
            env['YT_TOKEN'] = yt_token
            env['YT_PROXY'] = 'hahn.yt.yandex.net'
            env['YT_LOG_LEVEL'] = 'INFO'
            env['REQUESTS_CA_BUNDLE'] = '/etc/ssl/certs/'

            yesterday = date_to_str(datetime.datetime.now() - datetime.timedelta(days=1))
            start_date = self.Parameters.start_date or yesterday
            end_date = date_to_str(datetime.datetime.now())
            output_table = yesterday if start_date == yesterday else ('%s-%s' % (start_date, end_date))
            table_full_path = os.path.join(self.Parameters.output_dir, output_table)

            if self.Parameters.process_today_date:
                start_date = date_to_str(datetime.datetime.now())
                end_date = date_to_str(datetime.datetime.now() + datetime.timedelta(days=1))
                table_full_path = os.path.join(self.Parameters.output_dir, start_date)

            logging.info('Running...')
            execute_cmd(
                [
                    self.Parameters.get_phones_logs_executable_path,
                    '--output_table', table_full_path,
                    '--date_begin', start_date,
                    '--date_end', end_date,
                    '--api_password', sdk2.Vault.data(self.owner, 'uslugi_beeline_api_password'),
                    '--ydb_host', self.Parameters.ydb_host,
                    '--ydb_port', str(self.Parameters.ydb_port),
                    '--ydb_database', self.Parameters.ydb_database,
                    '--ydb_home', self.Parameters.ydb_home
                ],
                'ydo_get_phones_logs',
                'Failed to aggreagate logs',
                env=env
            )

        if self.Parameters.with_postprocess:
            with self.memoize_stage.postprocess:
                with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), "postprocess.sql")) as query_file:
                    query = query_file.read()
                task = RunYQL2(
                        self,
                        description='Postprocess proxy phones logs',
                        notifications=self.Parameters.notifications,
                        create_sub_task=False,
                        query=query,
                        custom_placeholders={'%GLOBAL_DATE%': datetime.datetime.utcnow().date().isoformat()},
                        trace_query=True,
                        use_v1_syntax=True,
                        publish_query=True,
                        yql_token_vault_name='YQL_TOKEN'
                )
                task.enqueue()
                raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

        logging.info('Done')
