import logging
import os
import tarfile
import tempfile

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.geosearch.CleanupYtFolder import clean_folder
from sandbox.projects.ydo.backup.LinkTables import YdoBackupLinkTables
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper
from sandbox.projects.ydo import (
    execute_cmd,
    get_now_utc,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment


class YdoProxyPhonesInspectorExecutable(sdk2.Resource):
    """
        Analyze logs and determine amount of proxy phones needed
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoProxyPhonesDistributorExecutable(sdk2.Resource):
    """
        Add and remove proxy phones according to workers' needs
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoProxyPhonesOffersSetter(sdk2.Resource):
    """
        Set proxy phones offers to workers
    """
    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoInspectDistributeProxyPhones(sdk2.Task):
    '''Task that analyzes logs, determines amount of proxy phones needed and set them for worker'''

    class Parameters(generate_view(
        inspector=YdoProxyPhonesInspectorExecutable,
        distributor=YdoProxyPhonesDistributorExecutable,
        geobase_snapshot=ResourceWrapper(GEODATA6BIN_STABLE),
        geodata_tz_snapshot=ResourceWrapper(GEODATATZDATA_STABLE),
        check_owner=True,
        with_environ=True,
        default_env_vars={
            'YT_PROXY': 'hahn.yt.yandex.net',
            'YT_LOG_LEVEL': 'INFO',
            'REQUESTS_CA_BUNDLE': '/etc/ssl/certs/',
        },
        default_secret_env_vars={
            'YT_TOKEN': 'yt-token',
            'YDB_TOKEN': 'ydb-token',
            'YQL_TOKEN': 'YQL_TOKEN',
            'TVM_SECRET': 'ydo_tvm_secret_prod',
        },
    )):
        yt_backup_path = sdk2.parameters.String('Path to YDB backup', required=True)

        yt_service_squeeze_path = sdk2.parameters.String('Path to squeezed search and order clicks dir', required=True)

        yt_distribution_log_path = sdk2.parameters.String('Path to distribution log dir', required=True)

        inspector_result_dir = sdk2.parameters.String('Path to inspector result dir', required=True)

        set_redirect_connections_threshold = sdk2.parameters.Integer('Worker\'s connections amount should be more than threshold to obtain proxy phone.', required=True, default=2)

        unset_redirect_connections_threshold = sdk2.parameters.Integer('Worker\'s connections amount should be less or equal than threshold to lose proxy phone.', required=True, default=0)

        max_phones_amount = sdk2.parameters.Integer('Max amount of proxy phones per worker', required=True, default=1)

        period_days = sdk2.parameters.Integer('Number of days for logs analysis', required=True, default=14)

        ydb_host = sdk2.parameters.String('YDB host', required=True)

        ydb_port = sdk2.parameters.Integer('YDB port', required=True, default=2135)

        ydb_database = sdk2.parameters.String('YDB database', required=True)

        ydb_home = sdk2.parameters.String('Path to YDB home dir', required=True)

        requests_timeout = sdk2.parameters.Float('Timeout for api requests', required=True, default=5.)

        max_to_delete = sdk2.parameters.Integer('Max amount of deleted proxy phones per one run', required=True, default=100)

        max_errors_cnt = sdk2.parameters.Integer('Max amount of errors per one run to be OK', required=True, default=100)

        telephony_statements_dir = sdk2.parameters.String('Path to telephony calls statements', required=True)

        unset_redirect_calls_threshold = sdk2.parameters.Integer('Worker\'s calls amount should be less or equal than threshold to lose proxy phone.', required=True, default=0)

        unset_period_days = sdk2.parameters.Integer('Minimal period while we do not unset proxy phones', required=True, default=14)

        caller_premedia_filename = sdk2.parameters.String('Premedia filename for caller side', default='')

        worker_premedia_filename = sdk2.parameters.String('Premedia filename for worker side', default='ydo_common_worker_premedia_human.wav')

        tvm_id = sdk2.parameters.Integer('Ydo tvm id. Need to use with telephony client', default=2002420)

        telephony_env = sdk2.parameters.String('Telephony environment', default='prod')

        async_api_config = sdk2.parameters.String('Development, testing, or production async API config', default='production')

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def inspect(self):
        logging.info('Started inspection of workers\' needs')
        self.Context.inspection_ts = get_now_utc()
        self.Context.inspector_out_table = os.path.join(self.Parameters.inspector_result_dir, '{}'.format(self.Context.inspection_ts))
        execute_cmd(
            [
                self.Parameters.inspector_path,
                '--ydb_backup_path', self.Parameters.yt_backup_path,
                '--service_squeeze_path', self.Parameters.yt_service_squeeze_path,
                '--set_redirect_connections_threshold', str(self.Parameters.set_redirect_connections_threshold),
                '--unset_redirect_connections_threshold', str(self.Parameters.unset_redirect_connections_threshold),
                '--max_phones_amount', str(self.Parameters.max_phones_amount),
                '--period_days', str(self.Parameters.period_days),
                '--geodata_path', self.Parameters.geobase_snapshot_path,
                '--result_path', self.Context.inspector_out_table, '--upload_to_yt',
                '--telephony_statements_dir', self.Parameters.telephony_statements_dir,
                '--unset_redirect_calls_threshold', str(self.Parameters.unset_redirect_calls_threshold),
                '--unset_period_days', str(self.Parameters.unset_period_days),
            ],
            'ydo_proxy_phones_inspector',
            'Failed to inspect logs for proxy phones needs',
            env=self.Parameters.get_environ(),
            fail_on_error=False,
        )

    def distribute(self):
        extracted_tzdata_dir = tempfile.mkdtemp()
        tarfile.open(self.Parameters.geodata_tz_snapshot_path).extractall(extracted_tzdata_dir)

        logging.info('Started distribution of proxy phones')
        execute_cmd(
            [
                self.Parameters.distributor_path,
                '--ydb_host', self.Parameters.ydb_host,
                '--ydb_port', str(self.Parameters.ydb_port),
                '--ydb_database', self.Parameters.ydb_database,
                '--ydb_home', self.Parameters.ydb_home,
                '--requests_timeout', str(self.Parameters.requests_timeout),
                '--max_to_delete', str(self.Parameters.max_to_delete),
                '--max_errors_cnt', str(self.Parameters.max_errors_cnt),
                '--inspector_result_path', self.Context.inspector_out_table,
                '--yt_distribution_log_path', self.Parameters.yt_distribution_log_path,
                '--max_phones_per_worker', str(self.Parameters.max_phones_amount),
                '--worker_premedia_filename', str(self.Parameters.worker_premedia_filename),
                '--caller_premedia_filename', str(self.Parameters.caller_premedia_filename),
                '--tvm_id', str(self.Parameters.tvm_id),
                '--telephony_env', str(self.Parameters.telephony_env),
                '--async_api_config', str(self.Parameters.async_api_config),
                '--geobase_file', self.Parameters.geobase_snapshot_path,
                '--geobase_tzdata_path', '{}/zones_bin'.format(extracted_tzdata_dir),
            ],
            'ydo_proxy_phones_distributor',
            'Failed to distribute proxy phones',
            env=self.Parameters.get_environ(),
            fail_on_error=False,
        )

    def link_inspector_result(self):
        logging.info('Linking last inspector result')
        yt_host = 'hahn'
        tables_to_link = {self.Context.inspector_out_table: os.path.join(self.Parameters.inspector_result_dir, 'current')}
        link_task = YdoBackupLinkTables(
            self,
            description='Link tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=yt_host,
            yt_vault_token='yt-token',
            yt_tables=tables_to_link
        )
        link_task.enqueue()

        raise sdk2.WaitTask([link_task.id], ctt.Status.Group.SUCCEED + ctt.Status.Group.SCHEDULER_FAILURE, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.inspection:
            self.inspect()

        with self.memoize_stage.distribution:
            self.distribute()

        with self.memoize_stage.cleaning:
            clean_folder(self, self.Parameters.inspector_result_dir, history_size=100)
            clean_folder(self, self.Parameters.yt_distribution_log_path, history_size=365)

        with self.memoize_stage.linking_to_current:
            self.link_inspector_result()

        logging.info('Done')
