import datetime
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.projects.ydo import ydo_releasers

from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE
from sandbox.projects.ydo import get_last_released_resource, execute_cmd, YdoFeaturesJson
from sandbox.projects.ydo.parameters_view import generate_view, with_config
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsDump
from sandbox.projects.ydo.resource_types import YdoDispatchPolygonRestrictionsJson



class YdoRemindersExpiredTasksCheckerExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoRemindersExpiredTasksChecker(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(generate_view(ydo_reminders_expired_tasks_checker=YdoRemindersExpiredTasksCheckerExecutable, check_owner=True, with_environ=True)):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('SAMSARA_CONNECTION'):
            samsara_host = sdk2.parameters.String('Samsara host', required=True)
        with sdk2.parameters.Group('YANG_CONNECTION'):
            yang_host = sdk2.parameters.String('Yang host', required=True)
        with sdk2.parameters.Group('BLACKBOX_CONNECTION'):
            blackbox_env = sdk2.parameters.String('Blackbox env', required=True)
            tvm_id = sdk2.parameters.String('Tvm id', required=True)
        with sdk2.parameters.Group('YDO_INTAPI_CONNECTION'):
            ydo_intapi_host = sdk2.parameters.String('Host', required=False)
        reminder_type_ids = sdk2.parameters.String('Reminder type ids', required=True)
        check_days = sdk2.parameters.String('Check days count', required=True, default=1)
        logging_config = sdk2.parameters.String('LoggingConfig')
        lookup_range_config = sdk2.parameters.String('Lookup range for pools in minutes', required=False)

        geobase_snapshot = sdk2.parameters.Resource(
            'Geobase 6 snapshot',
            resource_type=GEODATA6BIN_STABLE,
            required=True
        )

        geodata_tz_snapshot = sdk2.parameters.Resource(
            'Geodata tz',
            resource_type=GEODATATZDATA_STABLE,
            required=True
        )



    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        geodata_tz_location_path = str(
            get_last_released_resource(
                GEODATATZDATA_STABLE,
                condition=True,
                default=self.Parameters.geodata_tz_snapshot,
                error_msg="GEODATATZDATA_STABLE not found",
            ).path
        )
        extracted_tzdata_dir = tempfile.mkdtemp()
        tarfile.open(geodata_tz_location_path).extractall(extracted_tzdata_dir)

        rubrics_json_prod = get_last_resource(YdoMergedRubricsDump, 'stable')
        features_json_prod = get_last_resource(YdoFeaturesJson, 'stable')
        polygon_restrictions_json = get_last_resource(YdoDispatchPolygonRestrictionsJson, 'stable')

        cmd = [
            self.Parameters.ydo_reminders_expired_tasks_checker_path,
            '--kikimr_host', self.Parameters.ydb_host,
            '--kikimr_home', self.Parameters.ydb_home,
            '--kikimr_database', self.Parameters.ydb_database,
            '--kikimr_port', str(self.Parameters.ydb_port),

            '--samsara_host', self.Parameters.samsara_host,
            '--yang_host', self.Parameters.yang_host,

            '--blackbox_env', self.Parameters.blackbox_env,
            '--tvm_id', self.Parameters.tvm_id,

            '--rubrics_json', str(sdk2.ResourceData(rubrics_json_prod).path),
            '--features_json', str(sdk2.ResourceData(features_json_prod).path),
            '--polygon_restrictions_json', str(sdk2.ResourceData(polygon_restrictions_json).path),
            '--geobase_file', str(get_last_released_resource(
                GEODATA6BIN_STABLE,
                condition=True,
                default=self.Parameters.geobase_snapshot,
                error_msg="GEODATA6BIN_STABLE not found",
            ).path),
            '--geobase_tzdata_path', '{}/zones_bin'.format(extracted_tzdata_dir),

            '--reminder_type_ids', self.Parameters.reminder_type_ids,
            '--check_days', self.Parameters.check_days,
            '--ydo_intapi_host', self.Parameters.ydo_intapi_host,
            '--lookup_range_config', self.Parameters.lookup_range_config,
        ]

        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        self.add_param_to_cmd(cmd, '--kikimr_retry_count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr_delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr_backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr_limit', str(self.Parameters.ydb_limit))

        execute_cmd(cmd, 'ydo_reminders_expired_tasks_checker', 'ydo_reminders_expired_tasks_checker failed', env=self.Parameters.get_environ())


def get_last_resource(resource_type, release_status):
    last_resource = resource_type.find(attrs=dict(released=release_status)).first()
    if last_resource is not None:
        return last_resource
    return None
