import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoReplicationDiffProcessor(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/tools/replication/replication_diff_processor/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="replication_diff_processor")


class YdoSyncReplicationDiff(sdk2.Task, SolomonMixinV2):
    """Task that processes ERP employees schedule confirmations and sends notifications"""

    class Parameters(generate_view(
        processor=YdoReplicationDiffProcessor,
        check_owner=True,
    )):
        ydb_host = sdk2.parameters.String('YDB Host', required=True)
        ydb_port = sdk2.parameters.Integer('YDB Port', default=2135, required=True)
        ydb_database = sdk2.parameters.String('YDB Database', required=True)
        ydb_home = sdk2.parameters.String('YDB Home', required=True)
        ydb_token = sdk2.parameters.YavSecret("YDB token in yav", required=True)

        pg_host = sdk2.parameters.List('PostgreSQL hosts', required=True)
        pg_port = sdk2.parameters.Integer('PostgreSQL port', default=6432, required=True)
        pg_dbname = sdk2.parameters.String('PostgreSQL dbname', required=True)
        pg_user = sdk2.parameters.String('PostgreSQL user', required=True)
        pg_password = sdk2.parameters.YavSecret('PostgreSQL password in yav', required=True)
        pg_target_session_attrs = sdk2.parameters.String('PostgreSQL target_session_attrs', default='read-write', required=True)
        pg_sslmode = sdk2.parameters.String('PostgreSQL sslmode', default='require', required=True)
        pg_connect_timeout = sdk2.parameters.Integer('PostgreSQL connect_timeout', default=2, required=True)
        pg_use_random_host = sdk2.parameters.Bool('PostgreSQL use random host for connection', default=False)

        solomon_project = sdk2.parameters.String('Solomon project')
        solomon_cluster = sdk2.parameters.String('Solomon cluster')
        solomon_service = sdk2.parameters.String('Solomon service')
        solomon_label = sdk2.parameters.String('Solomon label')
        solomon_enabled = sdk2.parameters.Bool('Solomon enabled')
        solomon_token = sdk2.parameters.YavSecret('Solomon token in yav')

        batch_size = sdk2.parameters.Integer('Processing batch size [1; 1000]', default=50, required=True)
        pool_size = sdk2.parameters.Integer('Pool size', default=1, required=True)
        retries_on_diff = sdk2.parameters.Integer('Retries on diff', default=0, required=True)

        replication_async_api_env = sdk2.parameters.RadioGroup(
            'Replication Async Api Environment',
            choices=(
                ('production', 'production'),
                ('batch_production', 'batch_production'),
                ('testing', 'testing'),
            ),
            required=True,
            default='testing',
        )
        logbroker_token = sdk2.parameters.YavSecret("Logbroker token in yav", required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting replication data sync')

        env = os.environ.copy()
        env['YDB_TOKEN'] = self.Parameters.ydb_token.data()['YDB_TOKEN']
        env['LOGBROKER_TOKEN'] = self.Parameters.logbroker_token.data()['LOGBROKER_TOKEN']
        env['PG_PASSWORD'] = self.Parameters.pg_password.data()['PG_PASSWORD']

        args = [
            str(self.Parameters.processor_path),

            '--ydb_host', str(self.Parameters.ydb_host),
            '--ydb_port', str(self.Parameters.ydb_port),
            '--ydb_database', str(self.Parameters.ydb_database),
            '--ydb_home', str(self.Parameters.ydb_home),

            '--pg_host', ','.join(str(host) for host in self.Parameters.pg_host),
            '--pg_port', str(self.Parameters.pg_port),
            '--pg_dbname', str(self.Parameters.pg_dbname),
            '--pg_user', str(self.Parameters.pg_user),
            '--pg_target_session_attrs', str(self.Parameters.pg_target_session_attrs),
            '--pg_sslmode', str(self.Parameters.pg_sslmode),
            '--pg_connect_timeout', str(self.Parameters.pg_connect_timeout),

            '--replication-async-api-pipe-env', str(self.Parameters.replication_async_api_env),
            '--batch_size', str(self.Parameters.batch_size),
            '--pool-size', str(self.Parameters.pool_size),
            '--retries-on-diff', str(self.Parameters.retries_on_diff),
        ]

        if self.Parameters.pg_use_random_host:
            args.append('--pg_use_random_host')

        if self.Parameters.solomon_enabled:
            env['SOLOMON_TOKEN'] = self.Parameters.solomon_token.data()['SOLOMON_TOKEN']

            args.extend([
                '--solomon_project', str(self.Parameters.solomon_project),
                '--solomon_cluster', str(self.Parameters.solomon_cluster),
                '--solomon_service', str(self.Parameters.solomon_service),
                '--solomon_label', str(self.Parameters.solomon_label),
                '--solomon_enabled',
            ])

        execute_cmd(
            args,
            'sync_replication_diff',
            'Failed to sync replication diff',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoSyncReplicationDiff, self).on_finish(prev_status, status)
