import datetime
import os
import dateutil.parser
import logging

import sandbox.common.types.task as ctt

from sandbox import sdk2

from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.indexer.YdoRunYQL2 import YdoRunYQL2
from sandbox.projects.ydo.parameters_view import generate_view_for_yt
from sandbox.projects.ydo.resource_types import YdoResponseTimingsToSolomon
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


def date_range(start, end, step_days=1):
    step = datetime.timedelta(days=step_days)
    while start < end:
        yield start, min(start + step, end)
        start += step


def parse_iso_date(value):
    # type: (str) -> datetime.date
    return dateutil.parser.parse(value).date()


class YdoResponseTimeMonitoring(sdk2.Task, SolomonMixinV2):
    class Parameters(
        generate_view_for_yt(
            response_timings_to_solomon=YdoResponseTimingsToSolomon,
            check_owner=True,
            with_environ=True,
        )
    ):
        skip_push_to_solomon = sdk2.parameters.Bool("Skip pushing to solomon", default=False)
        with skip_push_to_solomon.value[False]:
            enable_tqdm = sdk2.parameters.Bool("Enable tqdm for logs", default=False)
        solomon_token = sdk2.parameters.YavSecret("Solomon token in yav", required=True)
        yql_token = sdk2.parameters.YavSecret("Yql token in yav", required=True)

        res_dir = sdk2.parameters.String("Resulting table directory", required=True)
        skip_yql_execution = sdk2.parameters.Bool("Skip all yqls", default=False)
        need_clean_yt_folder = sdk2.parameters.Bool("Need clean yt folder", default=True)
        yt_folder_size = sdk2.parameters.Integer("How many previous tables to leave", default=60)

        use_custom_date_interval = sdk2.parameters.Bool("Use custom date interval for calculate", default=False)
        with use_custom_date_interval.value[True]:
            start_date = sdk2.parameters.String("Calculate from date", required=True)
            end_date = sdk2.parameters.String("Calculate to date", required=True)

    def prepare_yql_task(self, date):
        # type: (datetime.date) -> None
        iso_date = date.isoformat()
        task = YdoRunYQL2(
            self,
            description="Calculate timings from http adapter",
            yql_script="calc_timings_from_http_adapter.yql",
            res_table_naming="custom",
            res_table_name=iso_date,
            start_date=iso_date,
            end_date=iso_date,
            need_create_table_link=False,
            need_clean_folder=self.Parameters.need_clean_yt_folder,
            yt_folder_size=self.Parameters.yt_folder_size,
            res_dir=self.Parameters.res_dir,
        )
        self.Context.run_script_task_ids.append(task.id)

    def get_date_inderval(self):
        # type: () -> tuple[str, str]
        start_date = getattr(self.Context, "start_date", None)
        end_date = getattr(self.Context, "end_date", None)
        if start_date and end_date:
            return start_date, end_date

        if self.Parameters.use_custom_date_interval:
            start_date = self.Parameters.start_date
            end_date = self.Parameters.end_date
        else:
            today = datetime.date.today()
            start_date = (today - datetime.timedelta(days=2)).isoformat()
            end_date = (today - datetime.timedelta(days=1)).isoformat()

        self.Context.start_date = start_date
        self.Context.end_date = end_date
        return start_date, end_date

    def gen_date_range(self, step_days=1):
        start_date, end_date = self.get_date_inderval()
        return date_range(parse_iso_date(start_date), parse_iso_date(end_date), step_days=step_days)

    def on_execute(self):
        run_script_task_ids = getattr(self.Context, "run_script_task_ids", None)
        if not run_script_task_ids and not self.Parameters.skip_yql_execution:
            self.Context.run_script_task_ids = []
            dates = tuple(self.gen_date_range())
            logging.info("all dates %s", dates)
            if len(dates) == 0:
                raise ValueError("End date less or equal start date")
            elif len(dates) > 60:
                raise ValueError("Too many days in interval. Maximum allowed 60 days")

            for date, _ in dates:
                self.prepare_yql_task(date)

        if not self.Parameters.skip_yql_execution:
            # Restarting sequentially performs all the prepared tasks
            for task_id in self.Context.run_script_task_ids:
                task = sdk2.Task[task_id]
                if task.status == ctt.Status.DRAFT:
                    task.enqueue()
                    raise sdk2.WaitTask([task.id], ctt.Status.Group.FINISH, wait_all=True)

        if not self.Parameters.skip_push_to_solomon:
            env = self.Parameters.get_environ()
            env["SOLOMON_TOKEN"] = self.Parameters.solomon_token.data()["SOLOMON_TOKEN"]
            env["YQL_TOKEN"] = self.Parameters.yql_token.data()["YQL_TOKEN"]
            for start_date, end_date in self.gen_date_range(step_days=5):
                start_date = start_date.isoformat()
                end_date = end_date.isoformat()
                cmd = [
                    self.Parameters.response_timings_to_solomon_path,
                    "--yt_dir", self.Parameters.res_dir,
                    "--start_date", start_date,
                    "--end_date", end_date,
                    "--solomon_token", "SOLOMON_TOKEN",
                ] + ([] if self.Parameters.enable_tqdm else ["--disable_tqdm"])
                execute_cmd(
                    cmd,
                    'ydo_push_response_timings_to_solomon__{}__{}'.format(start_date, end_date),
                    'Failed to push solomon timings to solomon',
                    env=env,
                )

    def on_break(self, *args, **kwargs):
        self.Parameters.save_info_about_resources()
        SolomonMixinV2.on_break(self, *args, **kwargs)

    def on_finish(self, *args, **kwargs):
        self.Parameters.save_info_about_resources()
        SolomonMixinV2.on_finish(self, *args, **kwargs)
