import logging
import os

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
    ydo_releasers,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.solomon_mixin import SolomonMixinV2


class YdoReturnBottlesProcessor(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = ydo_releasers
    arcadia_build_path = "ydo/tools/reminders/return_bottles/bin"
    executable = True

    binary_path = sdk2.parameters.String(default="return_bottles")


class YdoReturnBottles(sdk2.Task, SolomonMixinV2):
    """Task that creates samsara tickets for calls to return bottles"""

    class Parameters(
        generate_view(
            processor=YdoReturnBottlesProcessor,
            check_owner=True,
        )
    ):
        tokens = sdk2.parameters.YavSecret("YAV YDO_OAUTH_TOKENS secret id", required=True)

        ydb_host = sdk2.parameters.String('YDB Host', required=True)
        ydb_port = sdk2.parameters.Integer('YDB Port', default=2135, required=True)
        ydb_database = sdk2.parameters.String('YDB Database', required=True)
        ydb_home = sdk2.parameters.String('YDB Home', required=True)

        min_days = sdk2.parameters.Integer('Minimal number of days to call after', default=60, required=True)
        min_bottles_cnt = sdk2.parameters.Integer('Minimal number of bottles to call', default=4, required=True)

        reminders_backup_path = sdk2.parameters.String('Path to table reminders in YT backup', required=True)
        water_order_stats_path = sdk2.parameters.String('Path to table with water stats in YT', required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (PipEnvironment('requests'),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing return bottles processor')

        env = os.environ.copy()
        env['YDB_TOKEN'] = self.Parameters.tokens.data()['YDB_TOKEN']
        env['YQL_TOKEN'] = self.Parameters.tokens.data()['YQL_TOKEN']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                '--ydb_host',
                str(self.Parameters.ydb_host),
                '--ydb_port',
                str(self.Parameters.ydb_port),
                '--ydb_database',
                str(self.Parameters.ydb_database),
                '--ydb_home',
                str(self.Parameters.ydb_home),
                '--min_days',
                str(self.Parameters.min_days),
                '--min_bottles_cnt',
                str(self.Parameters.min_bottles_cnt),
                '--reminders_backup_path',
                str(self.Parameters.reminders_backup_path),
                '--water_order_stats_path',
                str(self.Parameters.water_order_stats_path),
            ],
            'return_bottles',
            'Failed to create return bottles tasks',
            env=env,
        )

        logging.info('Done')

    def on_finish(self, prev_status, status):
        SolomonMixinV2.on_finish(self, prev_status, status)
        super(YdoReturnBottles, self).on_finish(prev_status, status)
