# coding: utf-8
import logging
from time import time

import sandbox.sdk2 as sdk2
from sandbox.sandboxsdk import environments
from sandbox.projects.common import task_env
from sandbox.projects.ydo import ydo_releasers
from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class YdoSecretCustomerProcessorExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoSecretCustomerProcessor(sdk2.Task):
    """
        Обработка тайных покупателей в сервисной модели
    """

    class Requirements(task_env.TinyRequirements):
        ram = 1024
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(generate_view(ydo_secret_customer_processor=YdoSecretCustomerProcessorExecutable, check_owner=True, with_environ=True)):
        logging_config = sdk2.parameters.String('LoggingConfig')
        start_timestamp = sdk2.parameters.Integer('Start timestamp', default=0)
        smtp_token_name = sdk2.parameters.String('SMTP token name for robot-services', required=True)
        dry_run = sdk2.parameters.Bool('Dry run', default=False)

        ydo_robot_services_password = sdk2.parameters.YavSecret("Ydo robot_services password", required=True, default='sec-01ee83207yh22k0wfade3m19dg')

        with sdk2.parameters.RadioGroup("Run mode") as mode:
            mode.values["supply_quota"] = mode.Value("Supply secret customer order quota")
            mode.values["receive_feedback"] = mode.Value("Receive info via sending feedback form to customers")
            mode.values["send_feedback"] = mode.Value("Send aggregated feedback to vertical heads")
            mode.values["post_process"] = mode.Value("Post process answers")

        with mode.value["supply_quota"]:
            with sdk2.parameters.Group('KIKIMR_CONNECTION'):
                ydb_host = sdk2.parameters.String('Host', required=True)
                ydb_home = sdk2.parameters.String('Home', required=True)
                ydb_database = sdk2.parameters.String('Database', required=True)
                ydb_port = sdk2.parameters.Integer('Port', default=2135)
                ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
                ydb_delay = sdk2.parameters.Integer('Delay', default=4)
                ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
                ydb_limit = sdk2.parameters.Integer('Limit', default=15)

            oauth_token_name = sdk2.parameters.String('OAuth token name for robot-services', required=True)
            ab_api_host = sdk2.parameters.String('Ab host', choices=[('test', 'https://ab.test.yandex-team.ru'), ('prod', 'https://ab.yandex-team.ru')])
            prev_spec_table = sdk2.parameters.String('Path to prev_spec table', required=True)
            order_partner_table = sdk2.parameters.String('Path to order partner backup', required=True)
            spec_ratings_table = sdk2.parameters.String('Path to specialization ratings backup', required=True)
            total_budget = sdk2.parameters.String('Week budget for secret customer program', required=True)
            delay = sdk2.parameters.String('Delay between testid creation and supplying quota in db', default=15)

        with mode.value["receive_feedback"]:
            orders_table = sdk2.parameters.String('Path to orders backup', required=True)
            orders_aux_table = sdk2.parameters.String('Path to orders_aux backup', required=True)
            form_home = sdk2.parameters.String('Feedback form home url', required=True)
            backend_home = sdk2.parameters.String('Uslugi backend home url', required=True)
            feedback_table = sdk2.parameters.String('Path to feedback table', required=True)
            tvm_id = sdk2.parameters.Integer('TVM id', required=True)
            tvm_token_name = sdk2.parameters.String('TVM token env variable name', required=True)
            blackbox_env = sdk2.parameters.String('Blackbox env', choices=[('test', 'test'), ('mimino', 'mimino'), ('prod', 'prod')], required=True)

        with mode.value["post_process"]:
            categories_table = sdk2.parameters.String('Path to categories backup', required=True)
            feedback_table = sdk2.parameters.String('Path to feedback table', required=True)
            orders_table = sdk2.parameters.String('Path to orders backup', required=True)
            config_id = sdk2.parameters.String('Config id', choices=[('test', '/config_test'), ('prod', '/config_prod')], required=True)
            result_table = sdk2.parameters.String('Path to post processing result table', required=True)

        with mode.value["send_feedback"]:
            orders_table = sdk2.parameters.String('Path to orders backup', required=True)
            categories_table = sdk2.parameters.String('Path to categories backup', required=True)
            feedback_table = sdk2.parameters.String('Path to feedback table', required=True)
            config_id = sdk2.parameters.String('Config id', choices=[('test', '/config_test'), ('prod', '/config_prod')], required=True)

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, str(parameter)]

    def on_execute(self):
        cmd = [
            self.Parameters.ydo_secret_customer_processor_path,
        ]

        start_timestamp_value = str(self.Parameters.start_timestamp) if self.Parameters.start_timestamp else str(int(time()) - 24 * 60 * 60)
        self.add_param_to_cmd(cmd, '--start_timestamp', start_timestamp_value)
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        if self.Parameters.dry_run:
            cmd += ['--dry_run']

        cmd += [self.Parameters.mode]

        if self.Parameters.mode == 'supply_quota':
            self.add_param_to_cmd(cmd, '--kikimr-host', self.Parameters.ydb_host)
            self.add_param_to_cmd(cmd, '--kikimr-home', self.Parameters.ydb_home)
            self.add_param_to_cmd(cmd, '--kikimr-database', self.Parameters.ydb_database)
            self.add_param_to_cmd(cmd, '--kikimr-port', self.Parameters.ydb_port)
            self.add_param_to_cmd(cmd, '--kikimr-retry-count', self.Parameters.ydb_retry_count)
            self.add_param_to_cmd(cmd, '--kikimr-delay', self.Parameters.ydb_delay)
            self.add_param_to_cmd(cmd, '--kikimr-backoff', self.Parameters.ydb_backoff)
            self.add_param_to_cmd(cmd, '--kikimr-limit', self.Parameters.ydb_limit)
            self.add_param_to_cmd(cmd, '--delay', self.Parameters.delay)

        elif self.Parameters.mode == "send_feedback":
            self.add_param_to_cmd(cmd, '--smtp_token_name', self.Parameters.smtp_token_name)

        elif self.Parameters.mode == "receive_feedback":
            self.add_param_to_cmd(cmd, '--smtp_token_name', self.Parameters.smtp_token_name)

        self.add_param_to_cmd(cmd, '--ab_api_host', self.Parameters.ab_api_host)
        self.add_param_to_cmd(cmd, '--prev_spec_table', self.Parameters.prev_spec_table)
        self.add_param_to_cmd(cmd, '--total_budget', self.Parameters.total_budget)
        self.add_param_to_cmd(cmd, '--oauth_token_name', self.Parameters.oauth_token_name)
        self.add_param_to_cmd(cmd, '--orders_table', self.Parameters.orders_table)
        self.add_param_to_cmd(cmd, '--order_partner_table', self.Parameters.order_partner_table)
        self.add_param_to_cmd(cmd, '--spec_ratings_table', self.Parameters.spec_ratings_table)
        self.add_param_to_cmd(cmd, '--categories_table', self.Parameters.categories_table)
        self.add_param_to_cmd(cmd, '--feedback_table', self.Parameters.feedback_table)
        self.add_param_to_cmd(cmd, '--form_home', self.Parameters.form_home)
        self.add_param_to_cmd(cmd, '--backend_home', self.Parameters.backend_home)
        self.add_param_to_cmd(cmd, '--orders_aux_table', self.Parameters.orders_aux_table)
        self.add_param_to_cmd(cmd, '--tvm_id', self.Parameters.tvm_id)
        self.add_param_to_cmd(cmd, '--tvm_token_name', self.Parameters.tvm_token_name)
        self.add_param_to_cmd(cmd, '--blackbox_env', self.Parameters.blackbox_env)
        self.add_param_to_cmd(cmd, '--config_id', self.Parameters.config_id)
        self.add_param_to_cmd(cmd, '--result_table', self.Parameters.result_table)

        execute_cmd(cmd, 'ydo_secret_customer_processor', 'ydo_secret_customer_processor failed', env=self.Parameters.get_environ())
