import json
import os
import traceback

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import binary_task
from sandbox.projects.common.utils2 import resource_redirect_link
from sandbox.projects.ydo import (
    ydo_releasers,
    execute_cmd,
    YdoFeaturesJson,
)
from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.sandboxsdk import environments


class YdoUpgradeServiceCardsExecutable(sdk2.Resource):
    executable = True
    releaseable = True
    releasers = ydo_releasers


class YdoUpgradeServiceCardsReport(sdk2.Resource):
    ttl = 28


class YdoUpgradeServiceCards(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(
        generate_view(
            ydo_upgrade_service_cards_executable=YdoUpgradeServiceCardsExecutable,
            check_owner=True,
        )
    ):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('YDB settings') as ydb_settings_block:
            ydb_endpoint = sdk2.parameters.String('ydb_endpoint', required=True)
            ydb_database = sdk2.parameters.String('ydb_database', required=True)
            ydb_home = sdk2.parameters.String('ydb_home', required=True)

        with sdk2.parameters.Group('YT settings') as yt_settings_block:
            yt_proxy = sdk2.parameters.String('yt_proxy', default='hahn.yt.yandex.net')
            yt_dump_dir = sdk2.parameters.String('dump directory yt_path', required=True)

        rubricator_version = sdk2.parameters.Integer('rubricator_version', required=True)
        dry_run = sdk2.parameters.Bool('dry_run', required=True, default=True)
        worker_id = sdk2.parameters.String('worker_id')

        additional_cmd_params = sdk2.parameters.String('Additional cmd args')

        with sdk2.parameters.Output:
            report = sdk2.parameters.Resource(
                'Upgrade report',
                resource_type=YdoUpgradeServiceCardsReport,
            )

    class Requirements(sdk2.Requirements):
        cores = 1
        environments = (environments.PipEnvironment('jinja2', version='2.11.3'),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        env = os.environ.copy()
        env['YDB_TOKEN'] = sdk2.Vault.data(self.owner, 'ydb-token')
        env['YT_TOKEN'] = sdk2.Vault.data(self.owner, 'yt-token')
        env['YT_PROXY'] = self.Parameters.yt_proxy

        cmd_params = [
            self.Parameters.ydo_upgrade_service_cards_executable_path,
            '--kikimr_endpoint',
            self.Parameters.ydb_endpoint,
            '--kikimr_database',
            self.Parameters.ydb_database,
            '--kikimr_home',
            self.Parameters.ydb_home,
            '--db_dump_dir',
            self.Parameters.yt_dump_dir,
            '--dump_is_from_yt',
            '--rubricator_version',
            str(self.Parameters.rubricator_version),
            '--features_json',
            get_last_resource_data_path(YdoFeaturesJson, 'stable'),
            '--upgrade_dump_json',
            'upgrade_dump.json',
        ]
        if self.Parameters.dry_run:
            cmd_params.append('--dry_run')
        if self.Parameters.worker_id:
            cmd_params.extend(['--worker_id', self.Parameters.worker_id])
        if self.Parameters.additional_cmd_params:
            cmd_params.extend(self.Parameters.additional_cmd_params.split())

        exception = None
        try:
            execute_cmd(cmd_params, 'upgrade_service_cards', env=env)
        except TaskFailure as exception:
            self.set_info('Exception during execute_cmd\n{}'.format(traceback.format_exc()))
            self.set_info(open('error_dump', 'r').read())

        with open('upgrade_dump.json') as f:
            upgrade_dump = json.load(f)

        report_resource = self.render_report(upgrade_dump)
        self.Parameters.report = report_resource
        self.set_info(
            resource_redirect_link(report_resource.id, 'Upgrade report'),
            do_escape=False,
        )

        if exception is not None:
            raise exception

    def render_report(self, upgrade_dump):
        import library.python.resource as resource
        from jinja2 import Environment, BaseLoader
        env = Environment(loader=BaseLoader)
        env.policies['json.dumps_kwargs'] = {'ensure_ascii': False}
        template = env.from_string(resource.find(os.path.join(os.path.dirname(__file__), 'report_template.html')))
        rendered = template.render(
            worker_id_to_cards=upgrade_dump['worker_id_to_cards'],
            rubric_stats=upgrade_dump['rubric_stats'],
        )
        report_resource = YdoUpgradeServiceCardsReport(
            task=self,
            description='Upgrade report',
            path='resource.html',
        )
        report_data = sdk2.ResourceData(report_resource)
        with report_data.path.open('w') as f:
            f.write(rendered.decode('utf-8'))
        report_data.ready()
        return report_resource


def get_last_resource_data_path(resource_type, release_status):
    last_resource = resource_type.find(attrs=dict(released=release_status)).first()
    if last_resource is not None:
        return str(sdk2.ResourceData(last_resource).path)
    return None
