# -*- coding: utf-8 -*-
import logging
import os
import tempfile

from sandbox.sandboxsdk import environments
from sandbox import sdk2

from sandbox.projects.common.geosearch.utils import unpack_file

from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsSmallDump
from sandbox.projects.geobase.Geodata6BinStable.resource import GEODATA6BIN_STABLE
from sandbox.projects.resource_types import GEODATATZDATA_STABLE

from sandbox.projects.ydo.resource_types import (
    YdoGenerateServiceModelSitemapExecutable,
    YdoServiceModelCategoriesJson,
)

from sandbox.projects.ydo import execute_cmd


class YdoServiceModelUrls(sdk2.Resource):
    """File with urls"""


class YdoGenerateServiceModelSitemap(sdk2.Task):
    """Task that generate sm sitemap and upload it to s3"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 86400

        with sdk2.parameters.Group("Required resources") as required_resources_block:
            generate_service_model_sitemap_executable = sdk2.parameters.Resource(
                "generate sitemap executable",
                resource_type=YdoGenerateServiceModelSitemapExecutable,
                required=True,
            )

            s3_bucket = sdk2.parameters.String("s3 bucket", required=False)
            s3_secret = sdk2.parameters.YavSecret("s3 access key", required=True)
            rubrics_json = sdk2.parameters.Resource(
                'merged rubrics.json',
                resource_type=YdoMergedRubricsSmallDump,
                required=True,
            )

            geobase_snapshot = sdk2.parameters.Resource(
                'Geobase 6 snapshot',
                resource_type=GEODATA6BIN_STABLE,
                required=True,
            )

            geodata_tz_snapshot = sdk2.parameters.Resource(
                'Geodata tz',
                resource_type=GEODATATZDATA_STABLE,
                required=True,
            )

            service_model_categories_json = sdk2.parameters.Resource(
                'service model categories json',
                resource_type=YdoServiceModelCategoriesJson,
                required=True,
            )

            dry_run = sdk2.parameters.Bool(
                'do not upload to s3',
                default=False,
                required=False,
            )

    def run(self):
        logging.info("Running...")

        cmd = [
            str(sdk2.ResourceData(self.Parameters.generate_service_model_sitemap_executable).path),
            '--s3_bucket_name',
            self.Parameters.s3_bucket,
            '--s3_access_key_id',
            self.Parameters.s3_secret.data()["ydo_s3_access_key_id"],
            '--s3_secret_access_key',
            self.Parameters.s3_secret.data()["ydo_s3_secret_access_key"],
            '--rubrics_json',
            str(sdk2.ResourceData(self.Parameters.rubrics_json).path),
            '--geobase_path',
            str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path),
            '--service_model_categories_json',
            str(sdk2.ResourceData(self.Parameters.service_model_categories_json).path),
        ]

        if self.Parameters.dry_run:
            cmd.append('--dry_run')

        resource = YdoServiceModelUrls(self, "Urls", "root_dir")
        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)

        cmd = [str(arg) for arg in cmd]
        execute_cmd(
            cmd,
            "ydo_sm_sitemap_generator",
            "Failed to run sitemap generator",
            cwd=str(data.path),
        )
        data.ready()

    def on_execute(self):
        with self.memoize_stage.run:
            self.run()

        logging.info("Done")
