import datetime
import logging
import os
import json

import sandbox.sdk2 as sdk2

import sandbox.common.types.task as ctt
import sandbox.common.errors as ce

import sandbox.projects.common.solomon as lib_solomon

owner_config = {
    'YDO': {
        'token_name': 'solomon-token',
        'solomon_params': {
            'project': 'ydo',
            'cluster': 'sandbox_tasks',
            'service': 'simple_monitoring',
        },
    }
}


class SolomonMixin(object):
    def on_finish(self, prev_status, status, common_labels=(('host', 'sandbox-push'),)):
        try:
            config = owner_config[self.owner]
        except KeyError:
            logging.info('Unable get config')
            return

        try:
            solomon_token = sdk2.Vault.data(self.owner, config['token_name'])
        except (ce.VaultError, KeyError):
            logging.info('Unable receive solomon token')
            return

        try:
            if not isinstance(common_labels, dict):
                common_labels = dict(common_labels)
            common_labels.setdefault('host', 'sandbox-push')
            common_labels.setdefault('task', self.__class__.__name__)
            lib_solomon.push_to_solomon_v2(
                solomon_token,
                config['solomon_params'],
                generate_sensors(status == ctt.Status.SUCCESS),
                common_labels=common_labels,
            )
        except:
            logging.exception('Sensors was not been pushed')


class SolomonMixinV2(object):
    def get_custom_config(self):
        pass

    def get_common_labels(self):
        if not self.scheduler or self.scheduler < 0:  # < 0 manual runs
            return

        common_labels = dict(host='sandbox-push', scheduler=str(self.scheduler))
        return common_labels

    def send_metrics_to_solomon(self, status):
        common_labels = self.get_common_labels()
        if common_labels is None:
            return

        try:
            config = owner_config[self.owner]
        except KeyError:
            logging.info('Unable get config')
            return

        try:
            solomon_token = sdk2.Vault.data(self.owner, config['token_name'])
        except (ce.VaultError, KeyError):
            logging.info('Unable receive solomon token')
            return

        try:
            execution_time = int((datetime.datetime.now(self.created.tzinfo) - self.created).total_seconds())
            lib_solomon.push_to_solomon_v2(
                solomon_token,
                config['solomon_params'],
                generate_sensors_v2(status == ctt.Status.SUCCESS, execution_time),
                common_labels=common_labels,
            )
        except:
            logging.exception('Sensors was not been pushed')

        if os.path.exists('solomon_user_sensors.json'):
            try:
                with open('solomon_user_sensors.json') as solomon_user_sensors_file:
                    solomon_user_sensors = json.load(solomon_user_sensors_file)
                custom_config = self.get_custom_config() or config
                lib_solomon.push_to_solomon_v2(
                    solomon_token,
                    custom_config['solomon_params'],
                    solomon_user_sensors,
                    common_labels=common_labels,
                )
            except:
                logging.exception('User sensors was not been pushed')

    def on_break(self, prev_status, status):
        SolomonMixinV2.send_metrics_to_solomon(self, status)

    def on_finish(self, prev_status, status):
        SolomonMixinV2.send_metrics_to_solomon(self, status)


class SolomonByTaskTypeMixin(SolomonMixinV2):
    def get_common_labels(self):
        common_labels = dict(host='sandbox-push', task_type=str(self.type))
        return common_labels


def generate_sensors(has_succeeded):
    return [
        {'labels': {'sensor': 'execution'}, 'value': 1},
        {'labels': {'sensor': 'fail'}, 'value': 0 if has_succeeded else 1},
    ]


def generate_sensors_v2(has_succeeded, execution_time):
    return generate_sensors(has_succeeded) + [
        {
            'labels': {'sensor': 'execution_time_s'},
            'value': execution_time,
        },
    ]
