import json

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.ydo.parameters_view import generate_view
from sandbox.projects.ydo.default_multislot_task import YdoMutlihostTask
from sandbox.projects.ydo.universal_notifier import NotifySender
from sandbox.projects.ydo import ydo_releasers


class YdoCrmExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoNotifyRunner(YdoMutlihostTask):
    class Parameters(
        generate_view(
            check_owner=True,
            with_environ=True,
            default_secret_env_vars={
                "YDB_TOKEN": "ydb-token",
                "LOGBROKER_TOKEN": "LOGBROKER_TOKEN",
                "YQL_TOKEN": "YQL_TOKEN",
            },
        )

    ):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        json_query = sdk2.parameters.String("JSON config", default="", required=True)
        arcadia_config = sdk2.parameters.RadioGroup(
            'Config',
            choices=[
                ("production", "production/async_api_pipe_template.yaml"),
                ("testing", "testing/async_api_pipe_template.yaml"),
            ],
            default='production/async_api_pipe_template.yaml'
        )

    def _run_child_tasks(self, queries):
        subtasks = []
        for maillist in queries:
            if "query" not in maillist or "id" not in maillist:
                continue
            params = {
                "suffix": maillist.get("id"),
                "query": maillist.get("query"),
                "mail_id": maillist.get("mail_id"),
                "push_id": maillist.get("push_id"),
                "push_title": maillist.get("push_title"),
                "push_text": maillist.get("push_text"),
                "push_link": maillist.get("push_link"),
                "sms_text": maillist.get("sms_text"),
                "sms_link": maillist.get("sms_link"),
                "promocode": json.dumps(maillist.get("promocode", {})),
                "category": maillist.get("category", []),
                "scenario": maillist.get("scenario"),
            }

            sub_task = NotifySender.YdoNotifySender(
                self,
                description="Mailing {}".format(maillist.get("id")),
                owner=self.owner,
                notifications=self.Parameters.notifications,
                create_sub_task=False,
                ydb_host=self.Parameters.ydb_host,
                ydb_home=self.Parameters.ydb_home,
                ydb_database=self.Parameters.ydb_database,
                ydb_port=self.Parameters.ydb_port,
                ydb_retry_count=self.Parameters.ydb_retry_count,
                ydb_delay=self.Parameters.ydb_delay,
                ydb_backoff=self.Parameters.ydb_backoff,
                ydb_limit=self.Parameters.ydb_limit,
                secret_env=self.Parameters.secret_env,
                release_chooser=self.Parameters.release_chooser,
                arcadia_config=self.Parameters.arcadia_config,
                **params
            )
            subtasks.append(sub_task.enqueue())
        return subtasks

    def on_execute(self):
        with self.memoize_stage.create_children:
            queries = json.loads(str(self.Parameters.json_query))
            subtasks = self._run_child_tasks(queries)
            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, wait_all=True)
