import tempfile
from sandbox import sdk2

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils
from sandbox.projects.ydo import execute_cmd, YdoFeaturesJson

from sandbox.projects.ydo.parameters_view import generate_view, with_config
from sandbox.projects.ydo.default_multislot_task import YdoMutlihostTask
from sandbox.projects.ydo.resource_types import YdoUniversalNotifierExecutable
from sandbox.projects.ydo.rubrics_merger.YdoRubricsMerger import YdoMergedRubricsDump

class YdoNotifySender(YdoMutlihostTask):
    class Parameters(
        with_config(
            generate_view(
                ydo_universal_notifier=YdoUniversalNotifierExecutable,
                ydo_rubrics=YdoMergedRubricsDump,
                ydo_features=YdoFeaturesJson,
                check_owner=True,
                with_environ=True,
                default_secret_env_vars={
                    "YDB_TOKEN": "ydb-token",
                    "LOGBROKER_TOKEN": "LOGBROKER_TOKEN",
                    "YQL_TOKEN": "YQL_TOKEN",
                },
            ),
            "ydo/async_api/client_configs",
            [
                ("production", "production/async_api_pipe_template.yaml"),
                ("testing", "testing/async_api_pipe_template.yaml"),
            ],
        )
    ):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)

        with sdk2.parameters.Group("Util parameters") as util_block:
            suffix = sdk2.parameters.String("Suffix (id for universal)", default="", required=True)
            query = sdk2.parameters.String("Link to query", default="", required=True)
            mail_id = sdk2.parameters.String("Mail Id", required=False)
            push_id = sdk2.parameters.String("Push id", required=False)
            push_title = sdk2.parameters.String("Push title", required=False)
            push_text = sdk2.parameters.String("Push text", required=False)
            push_link = sdk2.parameters.String("Push_link", required=False)
            sms_text = sdk2.parameters.String("Sms text", required=False)
            sms_link = sdk2.parameters.String("Sms link", required=False)
            promocode = sdk2.parameters.String("Promocode config", required=False)
            category = sdk2.parameters.List("Category", required=False)
            scenario = sdk2.parameters.String("Scenario", required=False)
        logging_config = sdk2.parameters.String('LoggingConfig')
        logbroker_source = sdk2.parameters.String("Source for Logbroker", required=True, default="YdoNotifySender")

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        with open(self.Parameters.config_path) as f_config:
            raw_config = f_config.read()
        filled_config = raw_config.replace('{{ SOURCE }}', str(self.Parameters.logbroker_source))
        filled_config_tempfile = tempfile.NamedTemporaryFile()
        file_utils.write_file(filled_config_tempfile.name, filled_config)
        filled_config_tempfile.seek(0)
        cmd = []
        cmd += [
            self.Parameters.ydo_universal_notifier_path,
            '--kikimr-host', self.Parameters.ydb_host,
            '--kikimr-home', self.Parameters.ydb_home,
            '--kikimr-database', self.Parameters.ydb_database,
            '--kikimr-port', str(self.Parameters.ydb_port),
        ]
        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))
        self.add_param_to_cmd(cmd, '--config', filled_config_tempfile.name)
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)

        self.add_param_to_cmd(cmd, '--query', str(self.Parameters.query))
        self.add_param_to_cmd(cmd, '--suffix', str(self.Parameters.suffix))
        if self.Parameters.mail_id:
            self.add_param_to_cmd(cmd, '--mail-key', str(self.Parameters.mail_id))
        if self.Parameters.push_id:
            if not self.Parameters.push_text or not self.Parameters.push_link or not self.Parameters.push_title:
                eh.check_failed("Need all three parameters for push")
            self.add_param_to_cmd(cmd, '--push-id', str(self.Parameters.push_id))
            self.add_param_to_cmd(cmd, '--push-title', str(self.Parameters.push_title))
            self.add_param_to_cmd(cmd, '--push-text', str(self.Parameters.push_text))
            self.add_param_to_cmd(cmd, '--push-link', str(self.Parameters.push_link))
        if self.Parameters.sms_text:
            self.add_param_to_cmd(cmd, '--sms-text', str(self.Parameters.sms_text))
        if self.Parameters.sms_link:
            self.add_param_to_cmd(cmd, '--sms-link', str(self.Parameters.sms_link))
        if self.Parameters.promocode:
            self.add_param_to_cmd(cmd, '--promocode', str(self.Parameters.promocode))
        if self.Parameters.category:
            self.add_param_to_cmd(cmd, '--category', str(self.Parameters.category))
        if self.Parameters.scenario:
            self.add_param_to_cmd(cmd, '--scenario', str(self.Parameters.scenario))
        self.add_param_to_cmd(cmd, '--author', self.author)
        self.add_param_to_cmd(cmd, '--rubrics_json', str(self.Parameters.ydo_rubrics_path))
        self.add_param_to_cmd(cmd, '--features_json', str(self.Parameters.ydo_features_path))
        execute_cmd(cmd, 'universal_notifier', 'universal_notifier failed', env=self.Parameters.get_environ())

