import datetime

from sandbox import sdk2
import sandbox.common.errors as ce
from sandbox.projects.ydo import execute_cmd, ydo_releasers, YdoWorkersGroups
from sandbox.projects.ydo.parameters_view import generate_view


class YdoWorkerActionsNotifierExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoWorkerActionsNotifier(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(generate_view(
            ydo_worker_groups=YdoWorkersGroups,
            ydo_worker_actions_notifier=YdoWorkerActionsNotifierExecutable,
            check_owner=True,
            with_environ=True
    )):
        with sdk2.parameters.Group('KIKIMR_CONNECTION'):
            ydb_host = sdk2.parameters.String('Host', required=True)
            ydb_home = sdk2.parameters.String('Home', required=True)
            ydb_database = sdk2.parameters.String('Database', required=True)
            ydb_port = sdk2.parameters.Integer('Port', default=2135)
            ydb_retry_count = sdk2.parameters.Integer('RetryCount', default=5)
            ydb_delay = sdk2.parameters.Integer('Delay', default=4)
            ydb_backoff = sdk2.parameters.Integer('Backoff', default=2)
            ydb_limit = sdk2.parameters.Integer('Limit', default=15)
        with sdk2.parameters.Group('YT_TABLES'):
            yt_workers = sdk2.parameters.String('Workers', required=True)
            yt_puid_to_worker_id = sdk2.parameters.String('PuidToWorkerId', required=True)
            yt_service_cards = sdk2.parameters.String('ServiceCards', required=True)
            yt_categories = sdk2.parameters.String('Categories', required=True)
            yt_worker_settings = sdk2.parameters.String('WorkersSettings', required=True)
            yt_direct_info = sdk2.parameters.String('DirectInfo', required=True)
            yt_direct_operations = sdk2.parameters.String('DirectOperations', required=True)
            yt_achievements = sdk2.parameters.String('Achievements')
        with sdk2.parameters.Group('TVM'):
            tvm_id = sdk2.parameters.Integer('TVM id')
            tvm_token_name = sdk2.parameters.String('TVM token env variable name')
        from_date = sdk2.parameters.String('FromDate in a format "YYYY-mm-dd"')
        logging_config = sdk2.parameters.String('LoggingConfig')
        intapi_host = sdk2.parameters.String('IntApi host', choices=[('test', 'shared-dev.hamster.yandex.ru'), ('prod', 'ydo-internal.yandex.net')])

    def add_param_to_cmd(self, cmd, name, parameter):
        if parameter:
            cmd += [name, parameter]

    def on_execute(self):
        cmd = []
        self.add_param_to_cmd(cmd, '--logging-config', self.Parameters.logging_config)
        cmd += [
            self.Parameters.ydo_worker_actions_notifier_path, 'main',
            '--kikimr-host', self.Parameters.ydb_host,
            '--kikimr-home', self.Parameters.ydb_home,
            '--kikimr-database', self.Parameters.ydb_database,
            '--kikimr-port', str(self.Parameters.ydb_port),
            '--workers', self.Parameters.yt_workers,
            '--puid_to_worker_id', self.Parameters.yt_puid_to_worker_id,
            '--service_cards', self.Parameters.yt_service_cards,
            '--categories', self.Parameters.yt_categories,
            '--worker_settings', self.Parameters.yt_worker_settings,
            '--direct_info', self.Parameters.yt_direct_info,
            '--direct_operations', self.Parameters.yt_direct_operations,
        ]
        self.add_param_to_cmd(cmd, '--achievements', self.Parameters.yt_achievements)

        self.add_param_to_cmd(cmd, '--kikimr-retry-count', str(self.Parameters.ydb_retry_count))
        self.add_param_to_cmd(cmd, '--kikimr-delay', str(self.Parameters.ydb_delay))
        self.add_param_to_cmd(cmd, '--kikimr-backoff', str(self.Parameters.ydb_backoff))
        self.add_param_to_cmd(cmd, '--kikimr-limit', str(self.Parameters.ydb_limit))
        self.add_param_to_cmd(cmd, '--from-date', self.Parameters.from_date or datetime.datetime.utcnow().strftime('%Y-%m-%d'))
        self.add_param_to_cmd(cmd, '--tvm_id', str(self.Parameters.tvm_id) if self.Parameters.tvm_id else None)
        self.add_param_to_cmd(cmd, '--tvm_token_name', str(self.Parameters.tvm_token_name))
        self.add_param_to_cmd(cmd, '--intapi_host', str(self.Parameters.intapi_host))

        worker_groups = self.Parameters.ydo_worker_groups_resource
        if worker_groups is None:
            raise ce.TaskFailure('Unable find worker_groups')
        self.add_param_to_cmd(cmd, '--groups_file_path', self.Parameters.ydo_worker_groups_path)

        execute_cmd(cmd, 'worker_actions_notifier', 'worker_actions_notifier failed', env=self.Parameters.get_environ())
