from datetime import datetime, timedelta

from sandbox import sdk2

from sandbox.projects.ydo import execute_cmd, ydo_releasers
from sandbox.projects.ydo.parameters_view import generate_view_for_yt
from sandbox.projects.ydo.solomon_mixin import SolomonMixin


class YdoYtWorkerReducerBinary(sdk2.Resource):
    releasers = ydo_releasers
    executable = True
    releasable = True


class YdoWorkerClicksReducer(sdk2.Task, SolomonMixin):
    """ Accumulates clicks for workers for YDO. """
    class Parameters(generate_view_for_yt(
        ydo_worker_reducer_binary=YdoYtWorkerReducerBinary,
        check_owner=True,
        with_environ=True
    )):
        input_date = sdk2.parameters.RadioGroup(
            'Date',
            choices=(
                ('yesterday', 'yesterday'),
                ('custom', 'custom'),
            ),
            required=True,
            default='yesterday',
            sub_fields={'custom': ['custom_input_date']}
        )
        custom_input_date = sdk2.parameters.StrictString(
            'YYYY-MM-DD',
            regexp=r'\d{4}-\d{2}-\d{2}',
            required=True
        )
        input_path = sdk2.parameters.String(
            'Input node path',
            required=True,
            default='//user_sessions/pub/uslugi-redir-log/daily'
        )
        output_path = sdk2.parameters.String(
            'Output node path',
            required=True,
            default_value='//home/ydo/users/zmiecer/worker-clicks'
        )
        expiration_time = sdk2.parameters.Integer(
            'Expiration time of the output table in days',
            required=True,
            default=90
        )
        force = sdk2.parameters.Bool('Force rewrite')

        with sdk2.parameters.RadioGroup('Environment') as solomon_env:
            solomon_env.values['production'] = solomon_env.Value(value='Production')
            solomon_env.values['testing'] = solomon_env.Value(value='Testing', default=True)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_input_date(self):
        if self.Parameters.input_date == 'yesterday':
            yesterday = datetime.today() - timedelta(days=1)
            input_date = yesterday.strftime('%Y-%m-%d')
        elif self.Parameters.input_date == 'custom':
            input_date = self.Parameters.custom_input_date
        return input_date

    def on_execute(self):
        input_date = self.get_input_date()
        cmd = [
            self.Parameters.ydo_worker_reducer_binary_path,
            '--input_date', input_date,
            '--input_path', self.Parameters.input_path,
            '--output_path', self.Parameters.output_path,
            '--expiration_time', str(self.Parameters.expiration_time)  # sp.Popen can't work with int
        ]
        if self.Parameters.force:
            cmd.append('--force')

        execute_cmd(
            cmd=cmd,
            log_name='ydo_worker_reducer',
            error_message='Worker reducer failed',
            env=self.Parameters.get_environ()
        )

    def on_finish(self, prev_status, status):
        SolomonMixin.on_finish(self, prev_status, status, common_labels={'environment': self.Parameters.solomon_env})
        sdk2.Task.on_finish(self, prev_status, status)
