import os

from datetime import datetime

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects.ydo import execute_cmd
from sandbox.projects.ydo import ydo_releasers
from sandbox.projects.ydo.backend.YdoSaasSearchProxy.resource_types import YdoSaasSearchProxyWord2Vec
from sandbox.sandboxsdk import environments


class YdoCalcReactionsViewsDataExecutable(sdk2.Resource):
    releasable = True
    releasers = ydo_releasers


class YdoOrdersEmbeddings(sdk2.Resource):
    """
        Json with orders titles and texts embeddings
    """
    releasable = True
    releasers = ydo_releasers


class YdoCalcReactionsViewsDataTask(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        calc_reactions_views_data_executable_resource = sdk2.parameters.Resource(
            "calc_reactions_views_data_executable_resource",
            resource_type=YdoCalcReactionsViewsDataExecutable,
            required=True,
        )

        word2vec_model_resource = sdk2.parameters.Resource(
            "word2vec_model_resource",
            resource_type=YdoSaasSearchProxyWord2Vec,
            required=True,
        )

        states_path = sdk2.parameters.String("Path to directory with states")

        output_path = sdk2.parameters.String("Path to directory where to store ouput")

        db_dump_path = sdk2.parameters.String("Path to directory with db dump")

        squeeze_path = sdk2.parameters.String("Path to directory with squeezes about workers on orders")

    def on_execute(self):
        import yt.wrapper as yt

        yt_proxy = 'hahn.yt.yandex.net'
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        yt.config['proxy']['url'] = yt_proxy
        yt.config['token'] = yt_token

        env = os.environ.copy()
        env['YT_PROXY'] = yt_proxy
        env['YT_TOKEN'] = yt_token
        env['YQL_TOKEN'] = sdk2.Vault.data(self.owner, 'YQL_TOKEN')
        env['YT_LOG_LEVEL'] = 'DEBUG'

        last_state_path = max(yt.list(self.Parameters.states_path, absolute=True))
        new_state_path = os.path.join(self.Parameters.states_path, datetime.now().strftime('%s'))

        orders_embeddings_resource = sdk2.ResourceData(YdoOrdersEmbeddings(self, 'Ydo orders embeddings', 'orders_embeddings.json'))

        cmd = [
            str(sdk2.ResourceData(self.Parameters.calc_reactions_views_data_executable_resource).path),
            '--input', last_state_path,
            '--output_state', new_state_path,
            '--output_export', self.Parameters.output_path,
            '--dump_from_yt', self.Parameters.db_dump_path,
            '--workers_on_orders_squeeze', self.Parameters.squeeze_path,
            '--word2vec_model', str(sdk2.ResourceData(self.Parameters.word2vec_model_resource).path / 'text'),
        ]

        execute_cmd(cmd, "calc_reactions_views_data", "calc_reactions_views_data failed", env=env)
        orders_embeddings_resource.ready()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
