import logging

from sandbox import sdk2
from sandbox.projects.ydo import ydo_releasers, execute_cmd
from sandbox.projects.ydo.parameters_view import generate_view

logger = logging.getLogger(__name__)


# Resource types
class YdoYtToLbExecutable(sdk2.Resource):
    """
    Send data to Logbroker
    """

    executable = True
    releasable = True
    releasers = ydo_releasers


class YdoYtTableToLb(sdk2.Task):
    class Parameters(
        generate_view(
            ydo_yt_to_lb=YdoYtToLbExecutable,
            check_owner=True,
            with_environ=True,
            default_env_vars={'YT_PROXY': 'hahn'},
            default_secret_env_vars={'YT_TOKEN': 'yt-token', 'LOGBROKER_TVM_SECRET': 'ydo_tvm_secret_prod'},
        )
    ):
        yt_path = sdk2.parameters.String("Path to table to send.", required=True)
        logbroker_topic = sdk2.parameters.String("Logbroker topic for sending.", required=True)
        logbroker_batch_size = sdk2.parameters.Integer(
            "Logbroker messages quantity to write into topic", default=10, required=True
        )
        client_tvm_id = sdk2.parameters.Integer("TVM id", required=True)
        fail_on_exception = sdk2.parameters.Bool("Fail on exception", default=False)
        raise_on_writes_errors = sdk2.parameters.Bool("Raise on writes errors", default=False)
        add_message_timestamp = sdk2.parameters.Bool("Add send timestamp to message", default=False)
        timestamp_field = sdk2.parameters.String("Field name for timestamp in message")
        lb_cooldown = sdk2.parameters.Float("Time for Logbroker to cooldown")
        lb_use_gzip_compression = sdk2.parameters.Bool("Use GZIP compression", default=False)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        cmd = (
            [
                self.Parameters.ydo_yt_to_lb_path,
                '--yt_path',
                self.Parameters.yt_path,
                '--logbroker_batch_size',
                str(self.Parameters.logbroker_batch_size),
                '--client_tvm_id',
                str(self.Parameters.client_tvm_id),
                '--logbroker_topic',
                self.Parameters.logbroker_topic,
            ]
            + (['--fail_on_exception'] if self.Parameters.fail_on_exception else [])
            + (['--raise_on_writes_errors'] if self.Parameters.raise_on_writes_errors else [])
            + (
                ['--add_message_timestamp', '--timestamp_field', self.Parameters.timestamp_field]
                if self.Parameters.add_message_timestamp
                else []
            )
            + (['--lb_cooldown', str(self.Parameters.lb_cooldown)] if self.Parameters.lb_cooldown else [])
            + (['--lb_write_codec', '1'] if self.Parameters.lb_use_gzip_compression else [])
        )

        execute_cmd(cmd, 'yt_to_lb', 'Failed export yt data to Logbroker', env=self.Parameters.get_environ())
