# coding: utf-8

import requests
import logging
import json
import time

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.sandboxsdk import environments as sb_sdk_env


LOGGER = logging.getLogger('ydt')
WIKI_API = 'https://wiki-api.yandex-team.ru/_api/frontend'
WP_API = 'https://workplace.z.yandex-team.ru/api/workplace.services.CatalogService'


def get_environments():
    return [
        sb_sdk_env.PipEnvironment("startrek_client", version="1.7.0", use_wheel=True)
    ]


class YDTExceptions(Exception):
    pass


def extract_name(name):
    if '-' not in name:
        return name

    return name[name.index('-') + 1:]


def impact_verticals_parse():
    response = requests.post(
        url='{api_url}/{method}'.format(api_url=WP_API, method='listVerticals')
    )

    if response.status_code != 200:
        raise YDTExceptions(response.text)

    impact_verticals = {
        current['name']: current['weight'] for current in json.loads(response.content).get('objects', [])
    }

    return impact_verticals


class IssueObject(object):
    def __init__(self, downtime_yandex, ticket, name_vertical, name_service, impact_duration, impact_vertical):
        """
        CamelCase use because this is format need in Statface dataset.
        """
        self.ticket = ticket
        self.downtime_yandex = downtime_yandex
        self.name_vertical = name_vertical
        self.name_service = name_service
        self.impact_duration = impact_duration
        self.impact_vertical = impact_vertical


class SupportStatException(Exception):
    """
    Base exception for SB-Task SupportStat
    """


class DtMetrics(object):

    URL = 'https://st-api.yandex-team.ru/v2/myself'
    STAT_URL = 'https://stat.yandex-team.ru/_api/report/data'
    UA = 'ydt_metrics_task'

    @staticmethod
    def is_exist_field(issue, field):
        return field in issue.__dict__['_value']

    @staticmethod
    def get_tag(tags, tag_name):
        """
        :type tags: List[unicode str]
        :type tag_name: unicode str
        :rtype: str
        Example:
        >>> get_tag(['product:WEB', 'test: TEST', 'vertical:service'], 'product')
        <<< 'WEB'
        """

        for line in tags:
            if line.count(':') == 0 or line.count(':') > 1:
                continue
            k, v = line.split(':')
            if k == tag_name:
                if tag_name != 'product':
                    return v
                return extract_name(v)
        return 'Bad tags'

    @staticmethod
    def get_diff_date(d1, d2):
        from datetime import datetime
        """
        :type d1: float Begin Time
        :type d2: float EndT ime
        :rtype : float diff(end, begin)
        """
        timestamp_d1 = time.mktime(datetime.strptime(d1.split('.')[0], '%Y-%m-%dT%H:%M:%S').timetuple())
        timestamp_d2 = time.mktime(datetime.strptime(d2.split('.')[0], '%Y-%m-%dT%H:%M:%S').timetuple())
        return timestamp_d2 - timestamp_d1

    def __init__(self, queue, path_stat, filter, st_token, stat_token):
        from startrek_client import Startrek

        LOGGER.info('Start initialize...')

        client = Startrek(useragent=self.UA, base_url=self.URL, token=st_token)

        query = 'Queue: {queue} {filter}'.format(queue=queue, filter=filter)

        self._issues = client.issues.find(query, per_page=7000)
        self._path_stat = path_stat
        self._issue_with_dt = {}
        self._impact_verticals = impact_verticals_parse()
        self._stat_token = stat_token

    def calc_and_upload_all(self):
        LOGGER.info('Start computing...')

        for issue in self._issues:
            if self.is_exist_field(issue, 'ydt') and \
                    self.is_exist_field(issue, 'sreEndTime') and \
                    self.is_exist_field(issue, 'sreBeginTime'):
                date, time = issue['sreBeginTime'].split('T')
                date = '{} {}:00'.format(date, time[:time.index('.')][:-3])
                print(date)
                if date not in self._issue_with_dt:
                    self._issue_with_dt[date] = []
                LOGGER.info(
                    'Create issue object with key {key} and in date {date}.'.format(key=issue['key'], date=date))

                name_vertical = self.get_tag(issue['tags'], 'product')
                downtime_yandex = issue['ydt']

                if downtime_yandex == 0:
                    continue

                self._issue_with_dt[date].append(
                    IssueObject(
                        ticket=issue['key'],
                        downtime_yandex=downtime_yandex,
                        name_vertical=name_vertical,
                        name_service=self.get_tag(issue['tags'], 'service'),
                        impact_duration=self.get_diff_date(issue['sreBeginTime'], issue['sreEndTime']),
                        impact_vertical=self._impact_verticals[name_vertical]
                        if name_vertical in self._impact_verticals else 0
                    )
                )

        LOGGER.info('Success creating issue objects!')

        self.upload_to_stat()

    def upload_to_stat(self):
        LOGGER.info('Start aggregating and creating JSON.')

        json_to_upload = []
        for date in self._issue_with_dt:
            for issue_object in self._issue_with_dt[date]:
                json_to_upload.append(
                    {
                        'fielddate': date,
                        'ticket': issue_object.ticket,
                        'downtime_yandex': issue_object.downtime_yandex,
                        'name_vertical': issue_object.name_vertical,
                        'name_service': issue_object.name_service,
                        'impact_duration': issue_object.impact_duration,
                        'impact_vertical': issue_object.impact_vertical,
                    }
                )

        LOGGER.info('JSON Created!\nCount items: {count_items}'.format(count_items=len(json_to_upload)))

        response = requests.post(
            url=self.STAT_URL,
            headers={'Authorization': 'OAuth {}'.format(self._stat_token)},
            data={
                'name': self._path_stat,
                'scale': 'i',
                'json_data': json.dumps({'values': json_to_upload}),
                '_append_mode': 1
            }
        )

        LOGGER.info('Status code from statface: {}'.format(response.status_code))

        if response.status_code != 200:
            raise YDTExceptions(response.text)

        LOGGER.info('Data success uploading!')


class YdtMetrics(sdk2.Task):
    issues = {}

    class Requirements(sdk2.Requirements):
        disk_space = 1024
        environments = get_environments()
        client_tags = ~ctc.Tag.LINUX_XENIAL

    class Context(sdk2.Task.Context):
        config = {}
        url = 'https://st-api.yandex-team.ru/v2/myself'
        user_agent = 'sandbox/projects/ydt_metrics'
        per_page = 5000

    def on_execute(self):
        st_token = sdk2.Vault.data('epsilond1', 'ydt_base_token')
        stat_token = sdk2.Vault.data('epsilond1', 'ydt_token_stat')

        instance = DtMetrics(
            queue='SPI',
            filter='AND Updated: >= now() - 3d',
            path_stat='Adhoc/epsilond1/my_ydt',
            st_token=st_token,
            stat_token=stat_token,
        )

        instance.calc_and_upload_all()
