from __future__ import print_function

from sandbox import sdk2
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.common import task_env
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess, ProcessLog

import requests
from requests.packages.urllib3.util.retry import Retry
import requests.adapters

from concurrent import futures

import datetime
import itertools
import json
import logging
import os
import random


class SnapshotMeta(object):
    __slots__ = ["cluster", "segment", "snapshot_path"]

    def __init__(self, cluster, segment, snapshot_path):
        self.cluster = cluster
        self.segment = segment
        self.snapshot_path = snapshot_path


class PlayResults(object):
    __slots__ = ["cluster", "segment", "snapshots"]

    def __init__(self, cluster, segment, snapshots):
        """
        :type cluster: str
        :type segment: str
        :type snapshots: list[str]
        """
        self.cluster = cluster
        self.segment = segment
        self.snapshots = snapshots


_BUILD_TARGETS = [
    "yp/scheduler_simulator/environment",
    "yp/scheduler_simulator/simtool",
    "sandbox/projects/yp/RunYpAllocationMonitoring/allocation_stats",
]


class SolomonPusher(object):
    PUSH_URL = "https://api.solomon.search.yandex.net/api/v2/push"

    def __init__(self, token):
        self.session = requests.Session()
        self.session.verify = False

        retry = Retry(
            backoff_factor=0.3,
            total=10
        )
        self.session.mount(
            'http://',
            requests.adapters.HTTPAdapter(max_retries=retry),
        )
        self.session.mount(
            'https://',
            requests.adapters.HTTPAdapter(max_retries=retry),
        )
        if token is not None:
            self.session.headers['Authorization'] = 'OAuth {}'.format(token)

    def push_sensors(self, project, service, cluster, common_labels, sensors):
        solomon_packet = dict(
            commonLabels=common_labels,
            sensors=sensors,
        )
        params = dict(project=project, service=service, cluster=cluster)
        result = self.session.post(
            self.PUSH_URL,
            params=params,
            json=solomon_packet,
            timeout=60,
        )
        result.raise_for_status()


class RunYpAllocationMonitoring(YaMake2):
    """
    Monitor cluster full reallocation
    """

    class Parameters(YaMake2.Parameters):
        description = "Build and run YP Allocation monitoring"

        class Requirements(task_env.BuildRequirements):
            pass  # YPSUPPORT-492

        use_aapi_fuse = build_parameters.UseArcadiaApiFuse(default_value=True)
        use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApi(default_value=True)

        targets = sdk2.parameters.String(
            "Targets (semicolon separated)",
            required=True,
            default_value=";".join(_BUILD_TARGETS),
        )

        with parameters.Group("YT parameters") as yt_params:
            yt_token_secret_id = parameters.YavSecret(
                label="YT token secret id",
                required=True,
                description="secret should contain keys: yt_token",
                default="sec-01ec5jtcg3z83w2w10vgnpj27y",
            )
            yt_pool = parameters.String("Yt pool")
            yt_log_level = parameters.String(
                "YT_LOG_LEVEL",
                default="DEBUG",
            )
            yt_clusters = parameters.List(
                "YT clusters",
                required=True,
                default=["arnold", "hahn"]
            )
            yt_play_root = parameters.String(
                "YT play root",
                required=True,
            )

        with parameters.Group("YP parameters") as yp_params:
            yp_token_secret_id = parameters.YavSecret(
                label="YP token secret id",
                required=True,
                description="secret should contain keys: yp_token",
                default="sec-01ec5jtcg3z83w2w10vgnpj27y",
            )
            yp_clusters = parameters.List(
                "YP clusters",
                required=True,
                default=["man-pre", "sas-test"],
            )
            yp_segments = parameters.List(
                "YP node segments",
                required=True,
                default=["default"],
            )

        with parameters.Group("Solomon parameters") as statistics:
            solomon_do_push = parameters.Bool("Send metrics to solomon", default=False)
            solomon_project = parameters.String("Solomon project", default="yp")
            solomon_token_secret_id = parameters.YavSecret(
                label="Solomon token secret id",
                required=True,
                description="secret should contain keys: solomon_token",
                default="sec-01ec5jtcg3z83w2w10vgnpj27y",
            )

    def _process_results(self, output_dir, snapshot_report_tuples):
        """

        :type output_dir: str
        :type snapshot_report_tuples: typing.Iterable[(SnapshotMeta, futures.Future[str|None])]
        """
        failed_snapshots = []
        stats_binary = os.path.join(
            output_dir,
            "sandbox/projects/yp/RunYpAllocationMonitoring/allocation_stats",
            "allocation_stats",
        )
        for snapshot, reports_future in snapshot_report_tuples:
            try:
                reports_path = reports_future.result()
            except Exception as err:
                failed_snapshots.append(snapshot)
                logging.error("Failed getting result for snapshot {}: {}".format(snapshot, err))
                continue

            if reports_path is None:
                failed_snapshots.append(snapshot)
                continue

            output_path = os.path.join(reports_path, "statistics.json")
            cmd = [
                stats_binary,
                "--label", "segment", snapshot.segment,
                reports_path,
                output_path,
                "unassigned",
                "--sensor", "unassigned_pods",
                "fragmentation",
                "--sensor", "hole_filling"
            ]
            with ProcessLog(self, logger="stats_{}_{}".format(snapshot.cluster, snapshot.segment)) as pl:
                run = subprocess.Popen(
                    cmd,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                run.communicate()
                if run.returncode != 0:
                    failed_snapshots.append(snapshot)
                    continue

            try:
                with open(output_path) as f:
                    sensors = json.load(f)
            except Exception as err:
                logging.error(
                    "Failed parsing sensors at cluster={0.cluster}, segment={0.segment}: {1}".format(snapshot, err)
                )
                failed_snapshots.append(snapshot)
                continue

            utc_now_ts = (datetime.datetime.utcnow() - datetime.datetime(1970, 1, 1)).total_seconds()
            for sensor in sensors:
                sensor.setdefault("ts", utc_now_ts)

            logging.info("Pushing solomon sensors:\n{}".format(
                json.dumps(sensors, sort_keys=True, indent=4)
            ))

            if self.Parameters.solomon_do_push:
                solomon_push = SolomonPusher(self.Parameters.solomon_token_secret_id.data()["solomon_token"])
                solomon_push.push_sensors(
                    project="yp",
                    service="yp_allocation_monitoring",
                    cluster="yp-" + snapshot.cluster,
                    common_labels={},
                    sensors=sensors,
                )

        if failed_snapshots:
            for snapshot in failed_snapshots:
                logging.error("Failed processing: cluster={0.cluster}, segment={0.segment}".format(snapshot))
            raise RuntimeError("Failed to process some clusters")

    def post_build(self, source_dir, output_dir, pack_dir):
        super(RunYpAllocationMonitoring, self).post_build(source_dir, output_dir, pack_dir)
        logging.info("Running specific post build")

        snapshots = []
        for yp_cluster, yp_segment in itertools.product(self.Parameters.yp_clusters, self.Parameters.yp_segments):
            snapshots.append(
                SnapshotMeta(
                    cluster=yp_cluster,
                    segment=yp_segment,
                    snapshot_path="yt@{}@{}@latest".format(yp_cluster, yp_segment),
                )
            )
        snapshot_report_tuples = self._play_snapshots(source_dir, output_dir, snapshots)
        self._process_results(output_dir, snapshot_report_tuples)

    def _do_play_snapshot(self, yt_cluster, source_dir, output_dir, snapshot):
        """

        :type yt_cluster: str
        :type source_dir: str
        :type output_dir: str
        :type snapshot: SnapshotMeta
        :rtype: str|None
        """

        simulations_path = os.path.join(
            output_dir, "yp_simulation_{}_{}".format(snapshot.cluster, snapshot.segment)
        )
        os.mkdir(simulations_path)

        cmd = [
            os.path.join(output_dir, "yp/scheduler_simulator/simtool/simtool"),
            "--arcadia", output_dir,
            "--simulations-path", simulations_path,
            "run",
            "--experiment", os.path.join(
                source_dir,
                "yp/scheduler_simulator/data/experiments/allocation_monitoring.yson.j2"
            ),
            "--snapshot", snapshot.snapshot_path,
            "--sandbox-backend", "folder",
            "--skip-build",
            "--yt-play-on-cluster", yt_cluster,
            "--yt-play-root", self.Parameters.yt_play_root,
        ]
        try:
            with ProcessLog(self, logger="run_sim_on_{}_{}".format(snapshot.cluster, snapshot.segment)) as pl:
                env = os.environ.copy()
                env["YP_TOKEN"] = self.Parameters.yp_token_secret_id.data()["yp_token"]
                env["YT_TOKEN"] = self.Parameters.yt_token_secret_id.data()["yt_token"]
                env["YT_LOG_LEVEL"] = self.Parameters.yt_log_level
                if self.Parameters.yt_pool:
                    env["YT_POOL"] = self.Parameters.yt_pool
                run = subprocess.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
                run.communicate()
                if run.returncode != 0:
                    logging.warning("Run experiment failed failed with code: {}".format(run.returncode))
                    raise RuntimeError("OBLOM")
        except Exception:
            logging.error(
                "Simulation failed for snapshot: {}@{}".format(snapshot.cluster, snapshot.segment),
                exc_info=True
            )
            return None

        root = os.path.join(simulations_path, "runs")
        runs = os.listdir(root)
        assert len(runs) == 1
        run_reports = os.path.join(root, runs[0], "reports")
        return run_reports

    def _play_snapshots(self, source_dir, output_dir, snapshots):
        """

        :rtype: typing.Iterable[(SnapshotMeta, futures.Future[str|None])]
        """

        random.shuffle(snapshots)
        future_to_snapshot = {}
        first_play_by_clusters = {}
        yt_cluster_iterator = itertools.cycle(self.Parameters.yt_clusters)
        resource_upload_timeout = datetime.timedelta(minutes=15)
        with futures.ThreadPoolExecutor(max_workers=3 * len(self.Parameters.yt_clusters)) as tp:
            for snapshot in snapshots:
                yt_cluster = yt_cluster_iterator.next()
                if yt_cluster in first_play_by_clusters:
                    # NB: Let first run populate binaries cache in yt.
                    futures.wait([first_play_by_clusters[yt_cluster]], timeout=resource_upload_timeout.total_seconds())
                future = tp.submit(self._do_play_snapshot, yt_cluster, source_dir, output_dir, snapshot)
                first_play_by_clusters.setdefault(yt_cluster, future)
                future_to_snapshot[future] = snapshot
            for future in futures.as_completed(future_to_snapshot.keys()):
                yield future_to_snapshot[future], future

    def on_execute(self):
        super(RunYpAllocationMonitoring, self).on_execute()
