# coding=utf-8
from yp.analysis import fragmentation as analysis_fragmentation
from yp.analysis import metric
from yp.analysis import plot_hole_filling
from yp.analysis import yp_allocation

import click
import numpy

import json
import logging
import six


def configure_logging(debug):
    logging.basicConfig(
        format="%(asctime)-15s %(levelname)s %(message)s",
        level=logging.DEBUG if debug else logging.INFO,
    )


@click.group(chain=True)
@click.argument("src", type=click.Path())
@click.argument("dst", type=click.File(mode="w", atomic=True))
@click.option('--label', 'labels', type=click.Tuple([str, str]), multiple=True)
@click.option("--debug", is_flag=True)
def cli(src, dst, labels, debug):
    configure_logging(debug)


@cli.result_callback()
def process_pipeline(processors, src, dst, labels, debug):
    allocations = yp_allocation.load_experiment_yp_allocations(src)

    common_labels = {}
    for key, value in labels:
        assert key not in common_labels
        common_labels[key] = value

    result = []
    for processor in processors:
        result.extend(processor(allocations, common_labels))

    dst.write(json.dumps(result, sort_keys=True, indent=4))


@cli.command()
@click.option("--sensor", type=click.STRING)
@click.option('--label', 'labels', type=click.Tuple([str, str]), multiple=True)
def fragmentation(sensor, labels):
    local_labels = {}
    for label_key, label_value in labels:
        assert label_key not in local_labels
        local_labels[label_key] = label_value

    def processor(allocations, common_labels):
        sensors = []
        for allocation in allocations:
            dataframe_infos = []
            for dataframe in allocation.dataframes:
                median_pod = plot_hole_filling.get_pod_slice_median_hole(
                    dataframe.pods, [metric.ResourceKinds.CPU, metric.ResourceKinds.MEMORY]
                )
                dataframe_infos.append(
                    analysis_fragmentation.calculate_hole_filling(
                        dataframe,
                        median_pod,
                    )
                )
            percentiles = set(dataframe_infos[0].keys())
            hole_filling = {}
            for percentile in percentiles:
                hole_filling[percentile] = {}
                for resource in six.iterkeys(dataframe_infos[0][percentile]):
                    hole_filling[percentile][resource] = numpy.average([
                        x[percentile][resource] for x in dataframe_infos
                    ])

            additional_labels = common_labels.copy()
            additional_labels.update(local_labels)
            additional_labels["experiment"] = allocation.name
            sensors.extend(
                analysis_fragmentation.convert_to_solomon_format(
                    hole_filling,
                    sensor,
                    additional_labels,
                )
            )
        return sensors

    return processor


@cli.command()
@click.option("--sensor", type=click.STRING)
@click.option('--label', 'labels', type=click.Tuple([str, str]), multiple=True)
def unassigned(sensor, labels):
    local_labels = {}
    for label_key, label_value in labels:
        assert label_key not in local_labels
        local_labels[label_key] = label_value

    def processor(allocations, common_labels):
        sensors = []
        for allocation in allocations:
            additional_labels = common_labels.copy()
            additional_labels.update(local_labels)
            additional_labels["experiment"] = allocation.name
            additional_labels["sensor"] = sensor

            unassigned_avg = numpy.average([
                metric.get_nonassigned_pod_count(dataframe) for dataframe in allocation.dataframes
            ])
            sensors.append(
                dict(
                    value=unassigned_avg,
                    labels=additional_labels
                )
            )
        return sensors

    return processor


if __name__ == "__main__":
    cli()
